/*

    This code is part of Truss Solver which is free software.
    See the file COPYING for more details.

    Copyright (C) 1997  T. W. Shield, shield@aem.umn.edu

*/
#include "matrix.h"
#include <math.h>
#include <stdlib.h>


Matrix::Matrix(int size)
{
	init(size);
		
}

void Matrix::init(int size)
{

	int i;
	
	ndim = size;
	
/*
	data is ndim rows by (ndim+1) columns
	
	column ndim+1 is the rhs of the linear system Ax=b to solve
		
*/	
	data=(float *)malloc(ndim*(ndim+1)*sizeof(float));

	if(data == 0)
	{
		cout<<"out of memory in Matrix\n";
		exit(1);
	}
	
	for(i=0;i<ndim*(ndim+1);i++)
	{
		*(data+i)=0.;
	}
	
	have_rhs = 0; // no rhs defined yet -- for printing
		
}

Matrix::Matrix(Matrix &source) // copy constructor
{
int i,j;
	init(source.size()); 
	
	for(i=1;i<=ndim;i++)
	{
		for(j=1;j<=ndim;j++)
		{
			*(data+(i-1)*(ndim+1)+j-1)=source.array(i,j);
		}
	}
	
	if( source.rhs_def() == 1) // it has a rhs
	{
		for(i=1;i<=ndim;i++)
		{
			*(data+(i-1)*(ndim+1)+ndim)=source.array(i,ndim+1);
		}
		have_rhs=1;
	}
}

Matrix::Matrix(Matrix *source) // copy constructor from pointer
{
int i,j;
	init(source->size()); 
	
	for(i=1;i<=ndim;i++)
	{
		for(j=1;j<=ndim;j++)
		{
			*(data+(i-1)*(ndim+1)+j-1)=source->array(i,j);
		}
	}
	
	if( source->rhs_def() == 1) // it has a rhs
	{
		for(i=1;i<=ndim;i++)
		{
			*(data+(i-1)*(ndim+1)+ndim)=source->array(i,ndim+1);
		}
		have_rhs=1;
	}
		
}

Matrix::~Matrix()
{
	free(data);
	
}

void Matrix::enlarge(int inc) // adds inc rows and cols to matrix
{
int new_size;
int i,j;

	if(inc < 1)
		return;

	new_size=ndim+inc;
	
	data=(float *)realloc(data,new_size*(new_size+1)*sizeof(float));

	if(data == 0)
	{
		cout<<"out of memory in Matrix::enlarge\n";
		exit(1);
	}

	// transfer the data
	for(i=new_size;i > 0;i--)
	{
		for(j=new_size;j > 0;j--)
		{
		 
			if( i > ndim || j > ndim) // new data
			{
				*(data+(i-1)*(new_size+1)+j-1)=0.0;
			}
			else
			{
				*(data+(i-1)*(new_size+1)+j-1)=*(data+(i-1)*(ndim+1)+j-1);
			}
		}
	}
	
	ndim = new_size;
	
	// zero out the rhs
	for(i=1;i<=ndim;i++)
	{
		*(data+(i-1)*(ndim+1)+ndim) = 0.;
	}
	have_rhs = 0;
	
}

void Matrix::print(ostream *os)
{
int i,j;

	for(i=0;i<ndim;i++)
	{
		*os << "(";
		for(j=0;j<ndim;j++)
		{
			*os << *(data+i*(ndim+1)+j);
			if(j != ndim-1)
				*os << ",";
		}
		if( have_rhs == 1)
		{
			*os << ") ("<<*(data+i*(ndim+1)+ndim)<<")\n";
		}
		else
		{
			*os << ")\n";
		}
	}
}

ostream &operator<<(ostream &os, Matrix *m)
{
	m->print(&os);
	return(os);
}
ostream &operator<<(ostream &os, Matrix &m)
{
	m.print(&os);
	return(os);
}

void Matrix::operator=(Matrix source)
{
int i,j;

#ifdef DEBUG
	if( ndim != source.size())
	{
		cout << " size mismatch in Matrix::operator=\n";
		return;
	}
#endif

	for(i=1;i<=ndim;i++)
	{
		for(j=1;j<=ndim;j++)
		{
			*(data+(i-1)*(ndim+1)+j-1)=source.array(i,j);
		}
	}
	
	if( source.rhs_def() == 1) // it has a rhs
	{
		for(i=1;i<=ndim;i++)
		{
			*(data+(i-1)*(ndim+1)+ndim)=source.array(i,ndim+1);
		}
		have_rhs=1;
	}

}


Vector operator*(const Matrix &m,const Vector &v)
{
int i,j;
float sum;
Vector ans(m.size());

#ifdef DEBUG
	if( v.size() != m.size() )
	{
		cout << " size mismatch in Matrix::operator*:"<<v.size()<<"\n";
		return(ans);
	}
#endif


	for(i=1;i<=m.size();i++)
	{
		sum = 0.;
		for(j=1;j<=m.size();j++)
		{
			sum = sum + v.val(j)*m.array(i,j);
			//cout << v.val(j)<<"*"<<m.array(i,j)<< "+";
		}
		ans.val(i,sum);
		//cout <<"= "<<sum<<"\n";
	}
	
	return(ans);
}

float Matrix::array(int i,int j)
{
/* 
	call with row,col in range 1...ndim
	
	set j = ndim+1 to returns rhs -- needed for solve
*/	

#ifdef DEBUG
	if(i < 1 || i > ndim)
	{
		cout << " i range error in array "<<i<<"\n";
		return(0.0);
	}
	
	if(j < 1 || j > ndim+1) // allow rhs to be returned
	{
		cout << " j range error in array "<<j<<"\n";
		return(0.0);
	}

#endif
	return(*(data+(i-1)*(ndim+1)+j-1));
}

float Matrix::array(int i,int j) const
{
/* 
	call with row,col in range 1...ndim
*/	

#ifdef DEBUG
	if(i < 1 || i > ndim)
	{
		cout << " i range error in array "<<i<<"\n";
		return(0.0);
	}
	
	if(j < 1 || j > ndim+1) // allow rhs to be returned
	{
		cout << " j range error in array "<<j<<"\n";
		return(0.0);
	}

#endif
	return(*(data+(i-1)*(ndim+1)+j-1));
}

void Matrix::array(int i,int j,float val)
{
/*
	sets element (i,j) to val
	
	j = ndim + 1 is rhs
*/

#ifdef DEBUG
	
	if(i < 1 || i > ndim)
	{
		cout << " i range error in set array "<<i<<"\n";
		return;
	}
	
	if(j < 1 || j > ndim+1)
	{
		cout << " j range error in set array "<<j<<"\n";
		return;
	}
#endif

	*(data+(i-1)*(ndim+1)+j-1) = val;
	
}

void Matrix::array_add(int i,int j,float val)
{
/*
	adds val to element (i,j) 
	
	will not set rhs, use rhs_add() instead
*/

#ifdef DEBUG
	
	if(i < 1 || i > ndim)
	{
		cout << " i range error in set array "<<i<<"\n";
		return;
	}
	
	if(j < 1 || j > ndim)
	{
		cout << " j range error in set array "<<j<<"\n";
		return;
	}
#endif

	*(data+(i-1)*(ndim+1)+j-1) += val;
	
}

float Matrix::rhs(int i)
{
#ifdef DEBUG

	if(i < 1 || i > ndim)
	{
		cout << " i range error in rhs "<<i<<"\n";
		return(0.0);
	}
#endif

	return(*(data+(i-1)*(ndim+1)+ndim));
}	

void Matrix::rhs(Vector *vec)
{
/*
	sets the rhs to vec
*/
#ifdef DEBUG
	if(vec->size() != ndim)
	{
		cout << "rhs vector size mismatch \n";
		return;
	}
#endif
	
	int i;
	
	for(i=1;i<=ndim;i++)
	{
		*(data+(i-1)*(ndim+1)+ndim) = vec->val(i);
	}
	
	have_rhs = 1;
	
}	

void Matrix::rhs(int i,float val)
{
#ifdef DEBUG
	if(i < 1 || i > ndim)
	{
		cout << " i range error in set rhs "<<i<<"\n";
		return;
	}
#endif

	*(data+(i-1)*(ndim+1)+ndim) = val;
		
	have_rhs = 1;
	
}

void Matrix::rhs_add(int i,float val)
{
/*
	adds val to current rhs value
*/
#ifdef DEBUG
	if(i < 1 || i > ndim)
	{
		cout << " i range error in set rhs "<<i<<"\n";
		return;
	}
#endif

	*(data+(i-1)*(ndim+1)+ndim) += val;
		
	have_rhs = 1;
	
}

int Matrix::rhs_def()
{
	return(have_rhs);
}

int Matrix::size()
{
	return(ndim);
}

int Matrix::size() const
{
	return(ndim);
}

Vector Matrix::solve(int *ret_code)
{
/*

	NOTE: This destroys the matrix and rhs !!!!
	
	returns the answer vector
	
	return codes:
	
		0 = OK
		1 = zero pivot encountered
	
*/
/* sr!(),disp!(),nonode) STATIC
    SHARED con%
    '
    'solution of simultaneous equations by Guass-Jordan Reduction
    'partial pivoting is used
    'subroutine converted from Applied Numerical Methods For 
    'Digital Computation, 3rd edition, page 277
    '
    'n% = size of matrix = 3*nonode+con%-1= number of equations to solve
*/    
        
    Vector ans(ndim);
        
    Matrix *b = new Matrix(ndim);
    
   int m,k,i,j,jj;
   float big,ab,temp,denom;
   
    m = ndim + 1;
    
    // search for largest pivot
label6:
   k=m-1;
   
   if( k == 1 ) goto  label11;
    
    jj=1;
    
    big=fabs(array(1,1));
    
    for( i=2;i<=k;i++)
    {
        ab=fabs(array(i,1));
        if( (big-ab)< 0.0) 
        {
           big=ab; 
           jj=i;
        } 
//label8:
    }
    
// make decision on row interchange

    if((jj-1) != 0 )
    {
    	for(j=1;j<=m;j++)
    	{
        	temp=array(jj,j);
        	array(jj,j,array(1,j));
        	array(1,j,temp);
    	}
    }
    
//calculate the elements of the B matrix
label11:

    denom = array(1,1);
    
    if(fabs(denom) < ZERO_TOL)
    {
#ifdef DEBUG
	cout << " zero denom in matrix::solve \n";
#endif
    	if( ret_code != 0)
    		*ret_code = 1;
    	return(ans); // all zeros
    }
    	    
    for(j=2;j<=m;j++)
    {
    	for(i=2;i<=ndim;i++)
    	{
            b->array(i-1,j-1,array(i,j)-array(1,j)*array(i,1)/denom);
        }
    }
    
    for(j=2;j<=m;j++)
    {
        b->array(ndim,j-1,array(1,j)/denom);
    }
    
// reduce column counter by one
    m--;
    
//assign the B matrix to the name sr
    
    for(j=1;j<=m;j++)
    {
    	for(i=1;i<=ndim;i++)
    	{
    		array(i,j,b->array(i,j));
    	}
    }
    
//check to see if the sr matrix consists of just one column

    if( m > 1)  goto label6;
    
    delete b;
    
    for(i=1;i<=ndim;i++)
    {
    	ans.val(i,array(i,1));
    }
    
    have_rhs = 0;
    
    if( ret_code != 0)
    	*ret_code = 0; // OK!
    
    return(ans);
    
       
}

void Matrix::assem(Matrix subm,int n1,int n2, int ndof)
{
/*

	subm is the sub matrix for between nodes n1 and n2
	ndof is the number of degrees of freedom for each node
	
	subm must have size 2*ndof x 2*ndof
	
	n1 and n2 must be such that
	
	ndim >= n1*ndof and ndim >= n2*ndof
	
*/

int i,j,k1,k2;

#ifdef DEBUG
	
	if( 2*ndof != subm.size())
	{
		cout << "sub matrix size mismatch in Matrix::assem\n";
		return;
	}
	
	if( n1*ndof > ndim)
	{
		cout << "n1 out of range in Matrix::assem\n";
		return;
	}
	
	if( n2*ndof > ndim)
	{
		cout << "n2 out of range in Matrix::assem\n";
		return;
	}
	
#endif

//   locations in the main matrix
	
	k1 = (n1-1)*ndof;
	k2 = (n2-1)*ndof;	
		
	for(i=1;i<=ndof;i++)
	{
		for(j=1;j<=ndof;j++)
		{
			array_add(k1+i,k1+j,subm.array(i,j));
			array_add(k1+i,k2+j,subm.array(i,j+ndof));
			array_add(k2+i,k1+j,subm.array(i+ndof,j));
			array_add(k2+i,k2+j,subm.array(i+ndof,j+ndof));
		}
	}
}

void Matrix::symm()
{
/*
	makes the matrix symmetric by assuming that
	one of (i,j) or (i,j) is nonzero data and
	replacing the zero other one with that data.
*/
int i,j;

	for(i=1;i<=ndim;i++)
	{
		for(j=1;j<i;j++)
		{
			if( array(i,j) == 0.)
			{
				array(i,j,array(j,i));
			}
			else // assume (j,i) is zero
			{
				array(j,i,array(i,j));
			}
		}
	}
}	

void Matrix::triv_row(int row,float b)
{	
/*
	set up the trivial equation Xrow = b
	
	for displacement BC
*/	
int j;

#ifdef DEBUG
	if( row < 1 || row > ndim)
	{
		cout << "row invalid in Matrix::triv_row = "<<row<<"\n";
		return;
	}
#endif

	for(j=1;j<=ndim;j++)
	{
		array(row,j,0.0);
	}	
	
	array(row,row,1.0);
	
	rhs(row,b);
	
}
