/*

    This code is part of Truss Solver which is free software.
    See the file COPYING for more details.

    Copyright (C) 1997  T. W. Shield, shield@aem.umn.edu

*/

#include "model.h"

truss_model::truss_model()  // default constructor
{
	num_nodes = 0;
	num_elems = 0;
	num_mpsets = 0;

	node_xs=0;
	node_ys=0;
	connections=0;
	node_types=0;
	node_bcs=0;

	elem_node1=0;
	elem_node2=0;
	elem_mpset=0;
	elem_bcs=0;

	mpset_area=0;
	mpset_E=0;
	mpset_I=0;
	mpset_Q=0;
	mpset_SR1=0;
	mpset_SR2=0;
	mpset_NA_width=0;
	mpset_uses=0;
	
	num_loads=0;
	num_restraints=0;
	
	load_nodes=0;
	load_dirs=0;
	load_vals=0;
	
	restraint_nodes=0;
	restraint_dirs=0;
	restraint_vals=0;
	
	model_type = TRUSS; // default is truss (false)
	
	extra_rot_eqns = 0; // for solve
	rot_eqns_elem = 0;
	rot_eqns_node = 0;
	
	valid_solution = 0; // not solved yet

}

truss_model::~truss_model()  // default destructor
{
	num_nodes = 0;
	alloc_nodes(); // alloc_xxx does a free with num_xxx = 0
	
	num_elems = 0;
	alloc_elems();
	
	num_mpsets = 0;
	alloc_mpsets();
	
	num_loads = 0;
	alloc_loads();
	
	num_restraints = 0;
	alloc_restraints();
	
	discard_solution();
}

/****************************************************************

	input/output stream model functions
	
*****************************************************************/

#define TAB <<"\t"<<
#define TAB2 <<"\t\t"<<
#define ENDL <<"\n"

void truss_model::print(OSTREAM *os)
{
/*
	sends everything to output stream
*/

	write_model(os); // does nodes, mpsets, elements
	
	print_loads(os); // the loads
	
	print_restraints(os);// the restraints
	
	print_results(os); // the results
						
}

void truss_model::print_model(OSTREAM *os)
{
/*
	sends everything but the results to output stream
*/
	write_model(os); // does nodes, mpsets, elements
	
	print_loads(os); // the loads
	
	print_restraints(os);// the restraints

}

void truss_model::write_model(OSTREAM *os)
{
/*
	sends the model info: nodes, elements and mpsets
	to output stream
*/
int i;

	if( model_type == TRUSS)
		*os << "Truss Model\n\n";
	else
		*os << "Frame Model\n\n";
		
	*os << "Number of nodes = "<< num_nodes << "\n";
	*os << "Number of elems = "<< num_elems << "\n";
	*os << "Number of mpsets = "<< num_mpsets << "\n";

	*os << "\n";

	if(model_type == FRAME)
	{
	    *os <<"Mpset\tArea\tModulus\tI\tQ\tNA width\tSR1\tSR2\n";
	    for(i=0;i<num_mpsets;i++)
	    {
		*os << i+1<<"\t"<<mpset_area[i]<<"\t"<<mpset_E[i]<<"\t"<<mpset_I[i] <<"\t"<<mpset_Q[i]<<"\t"<<mpset_NA_width[i]<<"\t"<<mpset_SR1[i]<<"\t"<<mpset_SR2[i]<<"\n";
	    }
	}
	else
	{
	    *os <<"Mpset"<<"\t"<<"Area"<<"\t"<<"Modulus"<<"\n";
	    for(i=0;i<num_mpsets;i++)
	    {
		*os << i+1<<"\t"<<mpset_area[i]<<"\t"<<mpset_E[i]<<"\n";
	    }
	}
	*os << "\n";

	//*os <<"Node"<<"\t"<<"x coord"<<"\t"<<"y coord"<<"\t"<<"connects\n";
	*os <<"Node"<<"\t"<<"x coord"<<"\t"<<"y coord"<<"\n";
	
	for(i=0;i<num_nodes;i++)
	{
		//*os << i+1<<"\t"<<node_xs[i]<<"\t"<<node_ys[i]<<"\t"<<connections[i] <<"\n";
		*os << i+1<<"\t"<<node_xs[i]<<"\t"<<node_ys[i] << "\n";
	}
	*os << "\n";
	
	*os <<"Elem"<<"\t"<<"node 1"<<"\t"<<"node 2"<<"\t"<<"mpset\n";
	
	for(i=0;i<num_elems;i++)
	{
		*os << i+1<<"\t"<<elem_node1[i]+1<<"\t"<<elem_node2[i]+1<<"\t"<<elem_mpset[i]+1<<"\n";
	}
	*os << "\n";

}

void truss_model::print_loads(OSTREAM *os)
{
/*
	sends the loads information to output stream
*/
int i;
	*os << "Number of loads = "<< num_loads << "\n";
	*os <<"Load"<<"\t"<<"node/elem"<<"\t"<<"direction"<<"\t"<<"value\n";
	
	char temp[30];
	
	for(i=0;i<num_loads;i++)
	{
		load_dir_str(load_dirs[i],temp);
		*os << i+1<<"\t"<<load_nodes[i]+1<<"\t"<<temp<<"\t"<<load_vals[i]<<"\n";
	}
	*os << "\n";

}

void truss_model::print_restraints(OSTREAM *os)
{
/*
	sends the restraints info to the output stream
*/
int i;
	*os << "Number of restraints = "<< num_restraints << "\n";
	*os <<"Restraint"<<"\t"<<"node"<<"\t"<<"direction"<<"\t"<<"value\n";
	
	char temp[30];
		
	for(i=0;i<num_restraints;i++)
	{
		restraint_dir_str(restraint_dirs[i],temp);
		*os << i+1<<"\t"<<restraint_nodes[i]+1<<"\t"<<temp<<"\t"<<restraint_vals[i]<<"\n";
	}
	*os << "\n";
}

void truss_model::print_results(OSTREAM *os)
{
/*
	sends the results to the output stream
*/
int i;
	if(valid_solution == 1)
	{
		*os << "Displacements:" ENDL;
		if( model_type == TRUSS)
			*os << "Node\tu\tv\n";
		else
			*os << "Node\tu\tv\trotation\n";
		
		
		for(i=0;i<num_nodes;i++)
		{
			if( model_type == TRUSS)
				*os << i+1 TAB node_us[i] TAB node_vs[i] ENDL;
			else
				*os << i+1 TAB node_us[i] TAB node_vs[i] TAB node_rots[i] ENDL;
		}
		
		*os ENDL;
		
		*os << "Reaction Forces:" ENDL;
		
		if( model_type == TRUSS)
			*os << "Node" TAB "Dir" TAB "force" ENDL;
		else
			*os << "Node" TAB "Dir" TAB "force/moment" ENDL;
			
		char temp[30];
			
		for(i=0;i<num_restraints;i++)
		{
			load_dir_str(restraint_dirs[i],temp);
			*os << restraint_nodes[i]+1 TAB temp TAB restraint_forces[i] ENDL;
		
		}
		
		*os ENDL;
		
		*os << "Element Forces:" ENDL;
		
		if( model_type == TRUSS )
			*os << "Elem" TAB "Axial force" ENDL;
		else
			*os << "Elem" TAB "Axial force" TAB "Shear Force" TAB "Moment" ENDL;
		for(i=0;i<num_elems;i++)
		{
			if( model_type == TRUSS)
				*os << i+1 TAB elem_axial[i] ENDL;
		}
	}
}

OSTREAM &operator<<(OSTREAM &os, truss_model &truss)
{
/*
	output operator for the whole truss object
*/
	truss.print(&os);
	return os;
}

int truss_model::input_all(ISTREAM *is)
{
/*
	reads a model file (nodes, elements, mpsets, loads, restraints)
	
	returns OK, BAD, BAD_NODE or BAD_ELEM
	
	if returns not OK delete the object
*/
	if( empty() != 0) // only input on empty object
		return(BAD);
		
	if( read_model(is) != OK )
	{
		num_nodes=0; // make sure bad data is not used
		num_mpsets=0;
		num_elems=0;
		return(BAD);
	}

	if( read_loads(is) != OK )
	{
		num_nodes=0;// make sure bad data is not used
		num_mpsets=0;
		num_elems=0;
		return(BAD);
	}

	if( read_restraints(is) != OK )
	{
		num_nodes=0;// make sure bad data is not used
		num_mpsets=0;
		num_elems=0;
		return(BAD);
	}

	set_counters();

	return(OK);
}

int truss_model::input_loads(ISTREAM *is)
{
/*
	reads a loads file
	
	returns OK, BAD, BAD_NODE or BAD_ELEM
*/
	if( empty() == 0) // can't input on empty object
		return(BAD);
		
	valid_solution = 0;
		
	if( read_loads(is) != OK )
	{
		num_loads=0; // make sure bad data is not used
		alloc_loads();
		set_bc_counters();
		return(BAD);
	}
	
	set_bc_counters();

	return(OK);
}

int truss_model::input_restraints(ISTREAM *is)
{
/*
	read a restraints file
	
	returns OK, BAD, BAD_NODE or BAD_ELEM
*/
	if( empty() == 0) // can't input on empty object
		return(BAD);
		
	valid_solution = 0;
				
	if( read_restraints(is) != OK )
	{
		num_restraints=0; // make sure bad data is not used
		alloc_restraints();
		set_bc_counters();
		return(BAD);
	}
	
	set_bc_counters();

	return(OK);
}


void truss_model::set_counters()
{
/*
	go thru and set connections, mpset_used etc.

*/
	int i,n1,n2,mp;

	for(i=0;i<num_nodes;i++)
	{
		connections[i]=0;
	}	
	
	for(i=0;i<num_mpsets;i++)
	{
		mpset_uses[i]=0;
	}	

	for(i=0;i<num_elems;i++)
	{
		n1 = elem_node1[i];
		connections[n1]++;
		
		n2 = elem_node2[i];
		connections[n2]++;
		
		mp = elem_mpset[i];
		mpset_uses[mp]++;
	}
	
	set_bc_counters();
	
}

void truss_model::set_bc_counters()
{
/*
	set the node_bcs and elem_bcs counters
*/
int i,n1,d1;

	for(i=0;i<num_nodes;i++)
	{
		node_bcs[i]=0;
	}	
	
	for(i=0;i<num_elems;i++)
	{
		elem_bcs[i]=0;
	}	

	for(i=0;i<num_loads;i++)
	{
		n1 = load_nodes[i];
		
		d1 = load_dirs[i];
		
		if( d1 == DISTRIBUTED )
		{
			elem_bcs[n1]++;
		}
		else
		{
			node_bcs[n1]++;
		}
	}
	
	for(i=0;i<num_restraints;i++)
	{
		n1 = restraint_nodes[i];
		
		node_bcs[n1]++;
	}

}

void truss_model::getline(ISTREAM *is, char *line, int len)
{
#ifdef QT
QString qs;

	qs=is->readLine();

	strcpy(line,qs.left(len));
#else
	is->getline(line,len);	
#endif	
}
	
int truss_model::read_model(ISTREAM *is)
{
char line[100];

	getline(is,line,sizeof(line));
	
	model_type = -1;
	
	if( strstr(line,"Truss Model") != NULL )
		model_type = TRUSS;
		
	if( strstr(line,"Frame Model") != NULL )
		model_type = FRAME;
		
	if( model_type == -1)
	{
		return(BAD);
	}
	
	getline(is,line,sizeof(line)); // blank line
	
	getline(is,line,sizeof(line)); // num_nodes line
	if( parse_i(line,"Number of nodes = ",&num_nodes) == BAD)
		return(BAD);
	
	getline(is,line,sizeof(line)); // num_elems line
	if( parse_i(line,"Number of elems = ",&num_elems) == BAD)
		return(BAD);
	
	getline(is,line,sizeof(line)); // num_mpsets line
	if( parse_i(line,"Number of mpsets = ",&num_mpsets) == BAD)
		return(BAD);
	
	
	
/*
	need to malloc all the needed storage
*/		
	
	
	alloc_nodes();
	alloc_elems();
	alloc_mpsets();
	
			
	int i,ti,i1,i2,i3;
	float f1,f2;

	getline(is,line,sizeof(line)); // blank line

	if(model_type == FRAME)
	{
	    //*os <<"Mpset\tArea\tModulus\tI\tQ\tNA width\tSR1\tSR2\n";
	    //for(i=0;i<num_mpsets;i++)
	    //{
		//*os << i+1<<"\t"<<mpset_area[i]<<"\t"<<mpset_E[i]<<"\t"<<mpset_I[i] <<"\t"<<mpset_Q[i]<<"\t"<<mpset_NA_width[i]<<"\t"<<mpset_SR1[i]<<"\t"<<mpset_SR2[i]<<"\n";
	    //}
	}
	else
	{
	    //*os <<"Mpset"<<"\t"<<"Area"<<"\t"<<"Modulus"<<"\n";
	    
	    getline(is,line,sizeof(line)); // header line
	    if( strstr(line,"Mpset") == NULL)
	    	return(BAD);
	    	
	    for(i=0;i<num_mpsets;i++)
	    {
		//*os << i+1<<"\t"<<mpset_area[i]<<"\t"<<mpset_E[i]<<"\n";
	    	getline(is,line,sizeof(line)); // data line
	    	sscanf(line,"%i\t%f\t%f",&ti,&f1,&f2);
	    	mpset_area[i] = f1;
	    	mpset_E[i] = f2;		
		mpset_uses[i]=0; // computed below
	    }
	}
	getline(is,line,sizeof(line)); // blank line

	//*os <<"Node"<<"\t"<<"x coord"<<"\t"<<"y coord"<<"\t"<<"connects\n";
	getline(is,line,sizeof(line)); // header line
	if( strstr(line,"Node") == NULL)
	    	return(BAD);
	
	// missing read of node type for FRAME
	
	for(i=0;i<num_nodes;i++)
	{
		//*os << i+1<<"\t"<<node_xs[i]<<"\t"<<node_ys[i]<<"\t"<<connections[i] <<"\n";
	    	getline(is,line,sizeof(line)); // data line
	    	sscanf(line,"%i\t%f\t%f",&ti,&f1,&f2);
	    	node_xs[i] = f1;
	    	node_ys[i] = f2;
	    	node_types[i] = PIN;
	    	connections[i]=0; // computed below
	    	node_bcs[i]=0; // computed below
	}
	getline(is,line,sizeof(line)); // blank line
	
	//*os <<"Elem"<<"\t"<<"node 1"<<"\t"<<"node 2"<<"\t"<<"mpset\n";
	getline(is,line,sizeof(line)); // header line
	if( strstr(line,"Elem") == NULL)
	    	return(BAD);
	
	for(i=0;i<num_elems;i++)
	{
		//*os << i+1<<"\t"<<elem_node1[i]+1<<"\t"<<elem_node2[i]+1<<"\t"<<elem_mpset[i]+1<<"\n";
	    	getline(is,line,sizeof(line)); // data line
	    	sscanf(line,"%i\t%i\t%i\t%i",&ti,&i1,&i2,&i3);
	    	elem_node1[i] = i1-1;
	    	elem_node2[i] = i2-1;
	    	elem_mpset[i] = i3-1;
	    	elem_bcs[i] = 0;// computed below
	}
	getline(is,line,sizeof(line)); // blank line
	
	return(OK);
}

int truss_model::read_loads(ISTREAM *is)
{
int i,i1,ti;
float f1;
char line[100];

	getline(is,line,sizeof(line)); // num_loads line
	if( parse_i(line,"Number of loads = ",&num_loads) == BAD)
		return(BAD);
		
	alloc_loads();
		
	//*os <<"Load"<<"\t"<<"node/elem"<<"\t"<<"direction"<<"\t"<<"value\n";
	getline(is,line,sizeof(line)); // header line
	if( strstr(line,"Load") == NULL)
	    	return(BAD);
	
	//char temp[30];
	char fmt[50];
	
	for(i=0;i<num_loads;i++)
	{
		//load_dir_str(load_dirs[i],temp);
		//*os << i+1<<"\t"<<load_nodes[i]+1<<"\t"<<temp<<"\t"<<load_vals[i]<<"\n";
	    	getline(is,line,sizeof(line)); // data line
	    	
	    	if( strstr(line,"x direction") != NULL)
	    	{
	    		load_dirs[i]=X_DIRECTION;
	    		sprintf(fmt,"%%i\t%%i\tx direction\t%%f");
	    	}
	    	if( strstr(line,"y direction") != NULL)
	    	{
	    		load_dirs[i]=Y_DIRECTION;
	    		sprintf(fmt,"%%i\t%%i\ty direction\t%%f");
	    	}
	    	if( strstr(line,"moment") != NULL)
	    	{
	    		load_dirs[i]=MOMENT;
	    		sprintf(fmt,"%%i\t%%i\tmoment\t%%f");
	    	}
	    	if( strstr(line,"distributed") != NULL)
	    	{
	    		load_dirs[i]=DISTRIBUTED;
	    		sprintf(fmt,"%%i\t%%i\tdistributed\t%%f");
	    	}
	    	sscanf(line,fmt,&ti,&i1,&f1);
	    	
	    	if( load_dirs[i] != DISTRIBUTED)
	    	{
	    		if( i1 > num_nodes ) // out of range node number
	    			return(BAD_NODE);
	    	}
	    	else
	    	{
	    		if( i1 > num_elems ) // out of range elem number
	    			return(BAD_ELEM);
	    	}
	    	
	    	load_nodes[i]=i1-1;
	    	load_vals[i]=f1;
	}
	getline(is,line,sizeof(line)); // blank line
	
	return(OK);
}

int truss_model::read_restraints(ISTREAM *is)
{
int i,ti,i1;
float f1;
char line[100];

	getline(is,line,sizeof(line)); // num_restraints line
	if( parse_i(line,"Number of restraints = ",&num_restraints) == BAD)
		return(BAD);
		
	alloc_restraints();
		
	//*os <<"Restraint"<<"\t"<<"node"<<"\t"<<"direction"<<"\t"<<"value\n";
	getline(is,line,sizeof(line)); // header line
	if( strstr(line,"Restraint") == NULL)
	    	return(BAD);
	
	char fmt[50];
	
	for(i=0;i<num_restraints;i++)
	{
		//restraint_dir_str(restraint_dirs[i],temp);
		//*os << i+1<<"\t"<<restraint_nodes[i]+1<<"\t"<<temp<<"\t"<<restraint_vals[i]<<"\n";
	    	getline(is,line,sizeof(line)); // data line
	    	
	    	if( strstr(line,"x direction") != NULL)
	    	{
	    		restraint_dirs[i]=X_DIRECTION;
	    		sprintf(fmt,"%%i\t%%i\tx direction\t%%f");
	    	}
	    	if( strstr(line,"y direction") != NULL)
	    	{
	    		restraint_dirs[i]=Y_DIRECTION;
	    		sprintf(fmt,"%%i\t%%i\ty direction\t%%f");
	    	}
	    	if( strstr(line,"rotation") != NULL)
	    	{
	    		restraint_dirs[i]=ROTATION;
	    		sprintf(fmt,"%%i\t%%i\trotation\t%%f");
	    	}
	    	sscanf(line,fmt,&ti,&i1,&f1);
	    	
	    	if( i1 > num_nodes ) // out of range node number
	    		return(BAD_NODE);
	    		
	    	restraint_nodes[i]=i1-1;
	    	restraint_vals[i]=f1;
	}
	
	return(OK);
}

int truss_model::parse_i(char *line,char *test,int *ans)
{
/*
	returns OK or BAD
*/
char fmt[100];

	if( strstr(line,test) != NULL )
	{
		strcpy(fmt,test);
		strcat(fmt,"%i");
		sscanf(line,fmt,ans);
		return(OK);
	}
	
	return(BAD);
}

/****************************************************************

	overall model functions
	
*****************************************************************/
	
int truss_model::empty() // returns 1 if model is not empty
{
	if(num_nodes > 0)
		return(1);
		
	if(num_mpsets > 0)
		return(1);

	return(0);
}

int truss_model::type()
{
	return(model_type);
}

void truss_model::set_type(int type)
{
	if(empty() == 0 && (type == TRUSS || type == FRAME))
	{
		model_type = type;
	}
}

float truss_model::xmax()
{
float m,t;
	if(num_nodes == 0)
		return(1.0);

	m = -1.e6;

	for(int i=0;i<num_nodes;i++)
	{
		t = node_xs[i];
		if( t > m )
			m = t;
	}

	return(m);
}

float truss_model::ymax()
{
float m,t;
	if(num_nodes == 0)
		return(1.0);

	m = -1.e6;

	for(int i=0;i<num_nodes;i++)
	{
		t = node_ys[i];
		if( t > m )
			m = t;
	}

	return(m);
}

float truss_model::xmin()
{
float m,t;
	if(num_nodes == 0)
		return(0.0);

	m = 1.e6;

	for(int i=0;i<num_nodes;i++)
	{
		t = node_xs[i];
		if( t < m )
			m = t;
	}

	return(m);
}

float truss_model::ymin()
{
float m,t;
	if(num_nodes == 0)
		return(0.0);

	m = 1.e6;

	for(int i=0;i<num_nodes;i++)
	{
		t = node_ys[i];
		if( t < m )
			m = t;
	}

	return(m);
}

int truss_model::nodes()
{
	return(num_nodes);
}

int truss_model::elems()
{
	return(num_elems);
}

int truss_model::mpsets()
{
	return(num_mpsets);
}


/*******************************************************************

	Nodes
	
******************************************************************/

void truss_model::add_node(float x,float y)
{

	num_nodes++;
	
	alloc_nodes();
	
	node_xs[num_nodes-1]=x;
	node_ys[num_nodes-1]=y;
	connections[num_nodes-1]=0;
	node_types[num_nodes-1]=PIN;
	node_bcs[num_nodes-1]=0;
		
	discard_solution(); // model change invalidates soln

}

void truss_model::add_node(FPoint p)
{
	num_nodes++;
	
	alloc_nodes();
	
	node_xs[num_nodes-1]=p.x();
	node_ys[num_nodes-1]=p.y();
	connections[num_nodes-1]=0;
	node_types[num_nodes-1]=PIN;
	node_bcs[num_nodes-1]=0;
	
		
	discard_solution(); // model change invalidates soln

}

void truss_model::delete_node(int del) // del runs 1..num_nodes
{
	int i;

	if(del < 1 || del > num_nodes) // not a valid node
		return;
		
	if(connections[del-1] != 0 ) // still connected
		return;
		
	if(node_bcs[del-1] != 0) // still has load or restraint
		return;
		
	discard_solution(); // model change invalidates soln
		
	if(num_nodes == 1) // deleting the last node
	{
		num_nodes = 0;
		
		alloc_nodes(); // does a free
		
		return;
	}
		
	if(del != num_nodes)
	{	
	
		for(i=del;i<=num_nodes;i++)
		{
			node_xs[i-1]=node_xs[i];
			node_ys[i-1]=node_ys[i];
			node_types[i-1]=node_types[i];
			connections[i-1]=connections[i];
			node_bcs[i-1]=node_bcs[i];
		}
	
	}
		
	num_nodes--;
	
	alloc_nodes();
	
				
	for(i=0;i<num_elems;i++) // fix the element refs
	{
		if(elem_node1[i] >= del )
			elem_node1[i]--;
		if(elem_node2[i] >= del )
			elem_node2[i]--;
	}
			
	for(i=0;i<num_loads;i++) // fix the load refs
	{
		if(load_nodes[i] >= del )
			load_nodes[i]--;
	}
	
	for(i=0;i<num_restraints;i++) // fix the restraint refs
	{
		if(restraint_nodes[i] >= del )
			restraint_nodes[i]--;
	}
		
}

void truss_model::alloc_nodes()
{
/*
	sets the storage for the num_nodes for the node related 
	arrays.  
	
	does a free if num_nodes == 0
*/

    	if(num_nodes > 0)
    	{
		node_xs=(float *)realloc(node_xs,(num_nodes)*sizeof(float));
		if(node_xs == NULL)
			die();
		node_ys=(float *)realloc(node_ys,(num_nodes)*sizeof(float));
		if(node_ys == NULL)
			die();
		node_types=(int *)realloc(node_types,(num_nodes)*sizeof(int));
		if(node_types == NULL)
			die();
		connections=(int *)realloc(connections,(num_nodes)*sizeof(int));
		if(connections == NULL)
			die();
		node_bcs=(int *)realloc(node_bcs,(num_nodes)*sizeof(int));
		if(node_bcs == NULL)
			die();
	}
	else
	{
		free(node_xs);
		node_xs=0;
		free(node_ys);
		node_ys=0;
		free(node_types);
		node_types=0;
		free(connections);
		connections = 0;
		free(node_bcs);
		node_bcs=0;
	}
	
}


void truss_model::set_node_type(int node,int type)
{
	if(node < 1 || node > num_nodes)
		return;
		
	node_types[node-1]=type;
	
	discard_solution(); // model change invalidates soln
}
	
int truss_model::node_type(int node)
{
	if(node < 1 || node > num_nodes)
		return(-1);
	else	
		return(node_types[node-1]);
}	
	
float truss_model::node_x(int i) // i runs 1 to num_nodes
{
	if(i>0 && i<=num_nodes)
		return(node_xs[i-1]); 
	else
		return(0.0);
}

float truss_model::node_y(int i) // i runs 1 to num_nodes
{
	if(i>0 && i<=num_nodes)
		return(node_ys[i-1]); 
	else
		return(0.0);
}

int truss_model::connects(int i) // i runs 1 to num_nodes
{
	return(connections[i-1]);
}

void truss_model::set_node_pos(int i,float x,float y)
{
	if(i > 0 && i <= num_nodes)
	{
		node_xs[i-1]=x;
		node_ys[i-1]=y;
	
	}
	
	discard_solution(); // model change invalidates soln
}

int truss_model::coincident_node_chk(int n,FPoint p)
{
/*
	n = 1 ... num_nodes
	p = possible new node position
	
	checks to see if new node position 
	is too close to any other node
	
	returns:
		0 = ok
		1...nodes = index of node that is too close
*/
int i;
float xi,yi,tol,x,y,d;

	tol=NODE_CHK_TOL*(xmax()-xmin()); // slow ...
	
	x=p.x();
	y=p.y();

	for(i=0;i<num_nodes;i++)
	{
		if( i != n-1 ) // skip the checked node
		{
			xi=node_xs[i];
			yi=node_ys[i];
			d = sqrt((xi-x)*(xi-x)+(yi-y)*(yi-y));
		
			if(d < tol )
			{
				return(i+1); // return the too close node #
			}
		}			
	}
	
	return(0); // ok

}

void truss_model::set_node_pos(int i,FPoint p)
{
	if(i > 0 && i <= num_nodes)
	{
		node_xs[i-1]=p.x();
		node_ys[i-1]=p.y();
	}
	
	discard_solution(); // model change invalidates soln
}

FPoint truss_model::node(int i) // i runs 1 to nodes
{
	FPoint fp;
	if(i>0 && i<=num_nodes)
	{
		fp.setX(node_xs[i-1]);
		fp.setY(node_ys[i-1]);
	}
	return(fp);
}

/******************************************************

	Material Property Sets
	
*******************************************************/		

void truss_model::add_mpset_rod(float area,float E)
{
	num_mpsets++;
	
	alloc_mpsets();

	mpset_area[num_mpsets-1] = area;

	mpset_E[num_mpsets-1] = E;

	mpset_I[num_mpsets-1] = 1.0;
	
	mpset_Q[num_mpsets-1] = 1.0;
	
	mpset_SR1[num_mpsets-1] = 1.0;
	
	mpset_SR2[num_mpsets-1] = -1.0;
	
	mpset_NA_width[num_mpsets-1] = 1.0;
	
	mpset_uses[num_mpsets-1] = 0;

}

void truss_model::delete_mpset(int del) // del runs 1..num_mpsets
{
	int i;

	if(del < 1 || del > num_mpsets) // invalid mpset number
		return;
		
	if(mpset_uses[del-1] != 0 ) // still in use
		return;
		
	if(num_mpsets == 1) // deleting the last mpset
	{
		num_mpsets = 0;
		
		alloc_mpsets(); // does a free
		
		return;
	}
		
	if(del != num_mpsets)
	{	
	
		for(i=del;i<=num_mpsets;i++)
		{
			mpset_area[i-1]=mpset_area[i];
			mpset_E[i-1]=mpset_E[i];
			mpset_I[i-1]=mpset_I[i];
			mpset_Q[i-1]=mpset_Q[i];
			mpset_SR1[i-1]=mpset_SR1[i];
			mpset_SR2[i-1]=mpset_SR2[i];
			mpset_NA_width[i-1]=mpset_NA_width[i];
			
			mpset_uses[i-1]=mpset_uses[i];
		}
	
	}
		
	num_mpsets--;
	
	alloc_mpsets();
				
	for(i=0;i<num_elems;i++) // fix the element refs
	{
		if(elem_mpset[i] >= del )
			elem_mpset[i]--;
	}
				
}

void truss_model::alloc_mpsets()
{
/*
	allocates or reallocates the mpset related arrays
	
	does a free if num_mpsets == 0

*/	
	if(num_mpsets > 0)
	{
		mpset_area=(float *)realloc(mpset_area,(num_mpsets)*sizeof(float));
		if(mpset_area == NULL)
			die();
		mpset_E=(float *)realloc(mpset_E,(num_mpsets)*sizeof(float));
		if(mpset_E == NULL)
			die();
		mpset_I=(float *)realloc(mpset_I,(num_mpsets)*sizeof(float));
		if(mpset_I == NULL)
			die();
		mpset_Q=(float *)realloc(mpset_Q,(num_mpsets)*sizeof(float));
		if(mpset_Q == NULL)
			die();
		mpset_SR1=(float *)realloc(mpset_SR1,(num_mpsets)*sizeof(float));
		if(mpset_SR1 == NULL)
			die();
		mpset_SR2=(float *)realloc(mpset_SR2,(num_mpsets)*sizeof(float));
		if(mpset_SR2 == NULL)
			die();
		mpset_NA_width=(float *)realloc(mpset_NA_width,(num_mpsets)*sizeof(float));
		if(mpset_NA_width == NULL)
			die();
		mpset_uses=(int *)realloc(mpset_uses,(num_mpsets)*sizeof(int));
		if(mpset_uses == NULL)
			die();
	}
	else
	{
		free(mpset_area);
		mpset_area=0;
		free(mpset_E);
		mpset_E=0;
		free(mpset_I);
		mpset_I=0;
		free(mpset_Q);
		mpset_Q=0;
		free(mpset_SR1);
		mpset_SR1=0;
		free(mpset_SR2);
		mpset_SR2=0;
		free(mpset_NA_width);
		mpset_NA_width=0;
		free(mpset_uses);
		mpset_uses=0;		
	}
}
	
void truss_model::update_mpset_rod(int set,float area,float E)
{
/*
	updates the area and modulus, 
	
	only does the non-zero ones
	
*/
	if(set < 1 || set > num_mpsets)
		return;

	if( area != 0.)		
		mpset_area[set-1]=area;

	if( E != 0.)		
		mpset_E[set-1]=E;
	
	discard_solution(); // model change invalidates soln
	
}

void truss_model::update_mpset_beam(int set,float I,float Q,float SR1,float SR2,float NA_width)
{
/*
	only changes the non-zero parameters
	
*/
	if(set < 1 || set > num_mpsets)
		return;
		
	if( I != 0.)		
		mpset_I[set-1]=I;
	if( Q != 0.)		
		mpset_Q[set-1]=Q;
	if( SR1 != 0.)		
		mpset_SR1[set-1]=SR1;
	if( SR2 != 0.)		
		mpset_SR2[set-1]=SR2;
	if( NA_width != 0.)		
		mpset_NA_width[set-1]=NA_width;
		
	
	discard_solution(); // model change invalidates soln
	
}
	
float truss_model::area(int i) // i = 1..mpsets
{
	if(i > 0 && i <= num_mpsets)
		return(mpset_area[i-1]);
	else
		return(0.);		

}

float truss_model::modulus(int i) // i = 1..mpsets
{
	if(i > 0 && i <= num_mpsets)
		return(mpset_E[i-1]);
	else
		return(0.);		

}

float truss_model::I(int i) // i = 1..mpsets
{
	if(i > 0 && i <= num_mpsets)
		return(mpset_I[i-1]);
	else
		return(0.);		

}

float truss_model::Q(int i) // i = 1..mpsets
{
	if(i > 0 && i <= num_mpsets)
		return(mpset_Q[i-1]);
	else
		return(0.);		

}

float truss_model::SR1(int i) // i = 1..mpsets
{
	if(i > 0 && i <= num_mpsets)
		return(mpset_SR1[i-1]);
	else
		return(0.);		

}

float truss_model::SR2(int i) // i = 1..mpsets
{
	if(i > 0 && i <= num_mpsets)
		return(mpset_SR2[i-1]);
	else
		return(0.);		

}

float truss_model::NA_width(int i) // i = 1..mpsets
{
	if(i > 0 && i <= num_mpsets)
		return(mpset_NA_width[i-1]);
	else
		return(0.);		

}

int truss_model::mpset_used(int i) // i = 1..mpsets
{
	if(i > 0 && i <= num_mpsets)
		return(mpset_uses[i-1]);
	else
		return(0);		

}

/***********************************************************

	Elements
	
************************************************************/

int truss_model::add_elem(int n1,int n2, int mpset)
{
	if(mpset < 0 || mpset > num_mpsets ) // request for undef mpset
		return(BAD);

	if(n1 > num_nodes || n1 < 1) // undef node 1
		return(BAD);

	if(n2 > num_nodes || n2 < 1) // undef node 2
		return(BAD);
		
	if(check_duplicate(-1,n1-1,n2-1) != OK) // check against all current elems
		return(DUP);

	num_elems++;
	
	alloc_elems();
	
	elem_node1[num_elems-1] = n1-1;
	connections[n1-1]++;
	elem_node2[num_elems-1] = n2-1;
	connections[n2-1]++;
	elem_mpset[num_elems-1] = mpset-1;
	mpset_uses[mpset-1]++;
	elem_bcs[num_elems-1]=0;
	
	discard_solution(); // model change invalidates soln

	return(OK);
}

int truss_model::check_duplicate(int elem, int n1, int n2) // params go 0...n
{
/*
	private check routine for duplicate elements
	
	n1,n2 are node numbers starting at 0
	
	call with elem = -1 to check against all current elements
	
	returns OK or DUP
*/
	int i;
	
	for(i=0;i<num_elems;i++)
	{
		if(elem != i)
		{
		     if(n1 == elem_node1[i] && n2 == elem_node2[i])
			return(DUP);
			
		     if(n1 == elem_node2[i] && n2 == elem_node1[i])
			return(DUP);
		}
			
	}
	
	return(OK);
}

int truss_model::update_elem(int elem,int n1,int n2,int mpset)
{
	if(elem < 1 || elem > num_elems) // nonexistant element
		return(BAD); 

	if(num_mpsets < mpset) // request for undef mpset
		return(BAD);

	if(n1 > num_nodes || n1 < 1) // undef node 1
		return(BAD);

	if(n2 > num_nodes || n2 < 1) // undef node 2
		return(BAD);
		
	if(check_duplicate(elem-1,n1-1,n2-1) != OK) // duplicate element
		return(DUP);
		
	connections[elem_node1[elem-1]]--; // remove old connections
	connections[elem_node2[elem-1]]--;
	mpset_uses[elem_mpset[elem-1]]--;
		
	elem_node1[elem-1]=n1-1;
	connections[n1-1]++;
	elem_node2[elem-1]=n2-1;
	connections[n2-1]++;
	elem_mpset[elem-1]=mpset-1;
	mpset_uses[mpset-1]++;
	
	discard_solution(); // model change invalidates soln
	
	return(OK);
	
}	

void truss_model::delete_elem(int del) // del: 1 .. elems()
{
	int i;
	
	if(del < 1 || del > num_elems) // nonexistant element
		return; 
		
	if(elem_bcs[del-1] != 0) // still have distributed load
		return;
		
	connections[elem_node1[del-1]]--; // remove old connections
	connections[elem_node2[del-1]]--;
	mpset_uses[elem_mpset[del-1]]--;
		
	discard_solution(); // model change invalidates soln
	
	if(num_elems == 1) // deleting the last element
	{
		num_elems = 0;
		
		alloc_elems(); // does a free
		
		return;
	}
				
	if(del != num_elems)
	{	
		for(i=del;i<=num_elems;i++)
		{
			elem_node1[i-1]=elem_node1[i];
			elem_node2[i-1]=elem_node2[i];
			elem_mpset[i-1]=elem_mpset[i];
			elem_bcs[i-1]=elem_bcs[i];
		}
	
	}
	
	num_elems--;
	
	alloc_elems();
	
	for(i=0;i<num_loads;i++) // fix the load refs
	{
		if(load_dirs[i] == DISTRIBUTED) // then node is element number
		{
		 	if(load_nodes[i] >= del )
				load_nodes[i]--;
		}
	}
	
}

void truss_model::alloc_elems()
{
	if(num_elems > 0)
	{
		elem_node1=(int *)realloc(elem_node1,(num_elems)*sizeof(int));
		if(elem_node1 == NULL)
			die();
		elem_node2=(int *)realloc(elem_node2,(num_elems)*sizeof(int));
		if(elem_node2 == NULL)
			die();
		elem_mpset=(int *)realloc(elem_mpset,(num_elems)*sizeof(int));
		if(elem_mpset == NULL)
			die();
		elem_bcs=(int *)realloc(elem_bcs,(num_elems)*sizeof(int));
		if(elem_bcs == NULL)
			die();
	}
	else
	{
		free(elem_node1);
		elem_node1 = 0;
		free(elem_node2);
		elem_node2 = 0;
		free(elem_mpset);
		elem_mpset = 0;
		free(elem_bcs);
		elem_bcs = 0;			
	}
}

float truss_model::elem_x1(int i) // i is the element number 1 ... elems
{
	if(i > 0 && i <= num_elems) 
		return(node_xs[elem_node1[i-1]]);
	else
		return(0.0);
}

float truss_model::elem_y1(int i) // i is the element number 1 ... elems
{
	if(i > 0 && i <= num_elems) 
		return(node_ys[elem_node1[i-1]]);
	else
		return(0.0);
}

FPoint truss_model::elem_center(int i) // i runs 1 to elems
{
/*
	used for checking mouse clicks on main window
*/
	FPoint fp;
	if(i>0 && i<=num_elems)
	{
		fp.setX((node_xs[elem_node1[i-1]]+node_xs[elem_node2[i-1]])/2.);
		fp.setY((node_ys[elem_node1[i-1]]+node_ys[elem_node2[i-1]])/2.);
	}
	return(fp);
}


FPoint truss_model::elem_n1(int i) //i is the element number 1 ... elems
{
	FPoint fp;
	if(i > 0 && i <= num_elems)
	{
		fp.setX(node_xs[elem_node1[i-1]]);
		fp.setY(node_ys[elem_node1[i-1]]);
	}
	return(fp);
}

float truss_model::elem_x2(int i) // i is the element number 1 ... elems
{
	if(i > 0 && i <= num_elems)
		return(node_xs[elem_node2[i-1]]);
	else
		return(0.0);
}

float truss_model::elem_y2(int i) // i is the element number 1 ... elems
{
	if(i > 0 && i <= num_elems)
		return(node_ys[elem_node2[i-1]]);
	else
		return(0.0);
}

FPoint truss_model::elem_n2(int i) //i is the element number 1 ... elems
{
	FPoint fp;
	if(i > 0 && i <= num_elems)
	{
		fp.setX(node_xs[elem_node2[i-1]]);
		fp.setY(node_ys[elem_node2[i-1]]);
	}
	return(fp);
}

int truss_model::mpset(int i) //i is the element number 1 ... elems
{
/*
	returns mpset starting with 1 ...
*/
	if(i > 0 && i <= num_elems)
		return(elem_mpset[i-1]+1);
	else
		return(-1);
}

int truss_model::node1(int i) //i is the element number 1 ... elems
{
/*
	returns node1 starting with 1 ...
*/
	if(i > 0 && i <= num_elems)
		return(elem_node1[i-1]+1);
	else
		return(-1);
}

int truss_model::node2(int i) //i is the element number 1 ... elems
{
/*
	returns node2 starting with 1 ...
*/
	if(i > 0 && i <= num_elems)
		return(elem_node2[i-1]+1);
	else
		return(-1);
}

/**************************************************

	MISC FUNCTIONS
	
**************************************************/

void truss_model::example()
{
	add_node(1.,1.);
	add_node(3.,1.);
	add_node(2.,4.);

	add_mpset_rod(1.0,206.E3);

	if(add_elem(1,2,1) != OK) exit(1);	
	if(add_elem(2,3,1) != OK) exit(1);	
	if(add_elem(1,3,1) != OK) exit(1);	

	add_restraint(1,X_DIRECTION,0.);
	add_restraint(1,Y_DIRECTION,0.);
	add_restraint(2,Y_DIRECTION,0.);
	
	add_load(3,X_DIRECTION,1.0);
}

void truss_model::die()
{
	cout << "Memory allocation error in truss model\n";
	exit(1);
}

/**************************************************

	BOUNDARY CONDITIONS
	
**************************************************/

int truss_model::add_load(int item,int direction,float value)
{
/*
	item is a node number  unless direction is DISTRIBUTED
	then it is an element number
*/

	if(direction < 0 || direction > DISTRIBUTED)
		return(BAD);
		
	if(direction != DISTRIBUTED )
    	{
		if(item < 1 || item > num_nodes)
			return(BAD);
	}
	else
	{
		if(item < 1 || item > num_elems)
			return(BAD);
	}
		
	if(check_dup_bc(item,direction) != 0)
		return(DUP);
		
	num_loads++;
	
	alloc_loads();
		
	load_nodes[num_loads-1]=item-1;
	load_dirs[num_loads-1]=direction;
	load_vals[num_loads-1]=value;
	
	if(direction == DISTRIBUTED)
		elem_bcs[item-1]++;
	else
		node_bcs[item-1]++;
	
	discard_solution(); // model change invalidates soln
	
	return(OK);

}	

int truss_model::add_restraint(int node,int direction,float value)
{

	if(node < 1 || node > num_nodes)
		return(BAD);

	if(check_dup_bc(node,direction) != 0)
		return(DUP);
		
	if(direction < 0 || direction > ROTATION)
		return(BAD);
		
	num_restraints++;
	
	alloc_restraints();
		
	restraint_nodes[num_restraints-1]=node-1;
	restraint_dirs[num_restraints-1]=direction;
	restraint_vals[num_restraints-1]=value;
	
	node_bcs[node-1]++;

	discard_solution(); // model change invalidates soln
	
	return(OK);

}	

int truss_model::check_dup_bc(int node,int direction)
{
/*
	returns OK or DUP
*/
	int i;
		
	if(num_loads > 0)
	{
	    for(i=0;i<num_loads;i++)
	    {
		if((node-1) == load_nodes[i] && direction == load_dirs[i])
			return(DUP);
	     }
	}
	
	if(direction != DISTRIBUTED) // no distributed restraints
	{
	    if(num_restraints > 0)
	    {
		for(i=0;i<num_restraints;i++)
		{
			if((node-1) == restraint_nodes[i] && direction == restraint_dirs[i])
				return(DUP);
		}
	    }
	}
	
	return(OK);
	
}
	
void truss_model::delete_load(int item,int direction)
{
	int i,del;
	
	if(num_loads == 0)
		return;
		
	del = -1;
		
	for(i=0;i<num_loads;i++)
	{
		if((item-1) == load_nodes[i] && direction == load_dirs[i])
		{
			del = i+1;
			break;
		}
	}
	
	if(del == -1)
		return;
		
	discard_solution(); // model change invalidates soln
		
	if(direction == DISTRIBUTED) // update the counters
		elem_bcs[load_nodes[del-1]]--;
	else
		node_bcs[load_nodes[del-1]]--;

	if(num_loads == 1) // deleting the last load
	{
		num_loads = 0;
		
		alloc_loads(); // does a free
		
		return;
	}
	
	if(del != num_loads)
	{		
		for(i=del;i<=num_loads;i++)
		{
			load_nodes[i-1]=load_nodes[i];
			load_dirs[i-1]=load_dirs[i];
			load_vals[i-1]=load_vals[i];
		}
	
	}
	
	num_loads--;
	
	alloc_loads();
	
			
}	

void truss_model::alloc_loads()
{
	if(num_loads > 0)
	{
		load_nodes=(int *)realloc(load_nodes,(num_loads)*sizeof(int));
		if(load_nodes == NULL)
			die();
		
		load_dirs=(int *)realloc(load_dirs,(num_loads)*sizeof(int));
		if(load_dirs == NULL)
			die();
		
		load_vals=(float *)realloc(load_vals,(num_loads)*sizeof(float));
		if(load_vals == NULL)
			die();
	}
	else
	{			
		free(load_nodes);
		load_nodes = 0;
		free(load_dirs);
		load_dirs = 0;
		free(load_vals);
		load_vals = 0;
	}

}

void truss_model::delete_restraint(int item,int direction)
{
	int i,del;
	
	if(num_restraints == 0)
		return;
		
	del = -1;
		
	for(i=0;i<num_restraints;i++)
	{
		if((item-1) == restraint_nodes[i] && direction == restraint_dirs[i])
		{
			del = i+1;
			break;
		}
	}
	
	if(del == -1)
		return;
		
	discard_solution(); // model change invalidates soln
	
	node_bcs[restraint_nodes[del-1]]--; // update counters	

	if(num_restraints == 1) // deleting the last restraint
	{
		num_restraints = 0;
		
		alloc_restraints(); // does a free
		
		return;
	}
	
	if(del != num_restraints)
	{		
		for(i=del;i<=num_restraints;i++)
		{
			restraint_nodes[i-1]=restraint_nodes[i];
			restraint_dirs[i-1]=restraint_dirs[i];
			restraint_vals[i-1]=restraint_vals[i];
		}
	
	}
	
	num_restraints--;
	
	alloc_restraints();
	
}

void truss_model::alloc_restraints()
{
	if(num_restraints > 0)
	{
		restraint_nodes=(int *)realloc(restraint_nodes,(num_restraints)*sizeof(int));
		if(restraint_nodes == NULL)
			die();
		
		restraint_dirs=(int *)realloc(restraint_dirs,(num_restraints)*sizeof(int));
		if(restraint_dirs == NULL)
			die();
		
		restraint_vals=(float *)realloc(restraint_vals,(num_restraints)*sizeof(float));
		if(restraint_vals == NULL)
			die();
	}
	else
	{
		free(restraint_nodes);
		restraint_nodes = 0;
		free(restraint_dirs);
		restraint_dirs = 0;
		free(restraint_vals);
		restraint_vals = 0;
	}	
}

int truss_model::loads()
{
	return(num_loads);
}

int truss_model::restraints()
{
	return(num_restraints);	
}

int truss_model::node_in_bc(int node)
{
	if( node < 1 || node > num_nodes)
		return(0);
	else
		return(node_bcs[node-1]);
}

int truss_model::elem_in_bc(int elem)
{
	if( elem < 1 || elem > num_elems)
		return(0);
	else
		return(elem_bcs[elem-1]);
}

int truss_model::load_node(int n)
{
	if(n < 1 || n > num_loads)
		return(-1);
	else
		return(load_nodes[n-1]+1);

}

int truss_model::load_dir(int n)
{
	if(n < 1 || n > num_loads)
		return(-1);
	else
		return(load_dirs[n-1]);

}

float truss_model::load_val(int n)
{
	if(n < 1 || n > num_loads)
		return(0.);
	else
		return(load_vals[n-1]);

}

int truss_model::restraint_node(int n)
{
	if(n < 1 || n > num_restraints)
		return(-1);
	else
		return(restraint_nodes[n-1]+1);
}

int truss_model::restraint_dir(int n)
{
	if(n < 1 || n > num_restraints)
		return(-1);
	else
		return(restraint_dirs[n-1]);
}

float truss_model::restraint_val(int n)
{
	if(n < 1 || n > num_restraints)
		return(0.);
	else
		return(restraint_vals[n-1]);
}

void truss_model::load_dir_str(int i,char *ans)
{
	strcpy(ans,"error"); // in case of out of range
	
	if(i == X_DIRECTION)
		strcpy(ans,"x direction");
	if(i == Y_DIRECTION)
		strcpy(ans,"y direction");
	if(i == MOMENT)
		strcpy(ans,"moment");
	if(i == DISTRIBUTED)
		strcpy(ans,"distributed");
}

void truss_model::restraint_dir_str(int i,char *ans)
{
	strcpy(ans,"error"); // in case of out of range
	
	if(i == X_DIRECTION)
		strcpy(ans,"x direction");
	if(i == Y_DIRECTION)
		strcpy(ans,"y direction");
	if(i == MOMENT)
		strcpy(ans,"rotation");
}		

/**************************************************

	PROBLEM SOLUTION
	
**************************************************/
	
void truss_model::bar_geo(float *length,float *cosine,float *sine,float *delx,float *dely,int elem) 
{
/*
	returns some geometrical quantities about the 
	orientation of the bar in space
	
	elem = 1 ... num_elems
*/
    *delx=elem_x2(elem)-elem_x1(elem);
    *dely=elem_y2(elem)-elem_y1(elem);
    *length=sqrt(*delx*(*delx)+*dely*(*dely));
    *cosine=*delx/(*length);
    *sine=*dely/(*length);
}

Matrix truss_model::elem_stiff(int elem)
{
/*
	returns the element stiffness matrix for element elem
	
	elem = 1 ... num_elems
*/	
float length,cosine,sine,delx,dely,EI,EA;
float l2,l3,c2,s2;

Matrix estiff(6); 

	if( elem < 1 || elem > num_elems)
		return(estiff); // which is all zeros at this point

        bar_geo(&length,&cosine,&sine,&delx,&dely,elem);
        
        EA = modulus(mpset(elem))*area(mpset(elem));
        
        if(model_type == TRUSS)
        	EI = 0.;
        else
        	EI = modulus(mpset(elem))*I(mpset(elem));
        	
        l2=length*length;
        l3=l2*length;
        s2=sine*sine;
        c2=cosine*cosine;

/*
        setting up stiffness matrix for single bar
*/
	// fill the 1,1 block
        estiff.array(1,1,12.*s2*EI/l3+EA*c2/length);
        estiff.array(2,1,-12.*cosine*sine*EI/l3+EA*sine*cosine/length);
        estiff.array(3,1,-6.*sine*EI/l2);
        estiff.array(2,2,12.*c2*EI/l3+s2*EA/length);
        estiff.array(3,2,6.*cosine*EI/l2);
        estiff.array(3,3,4.*EI/length);
        
        // symmeterize -- done later
        //U!(1,2,end1%,end1%)=U!(2,1,end1%,end1%)
        //U!(1,3,end1%,end1%)=U!(3,1,end1%,end1%)
        //U!(2,3,end1%,end1%)=U!(3,2,end1%,end1%)
        
        // fill the 2,1 block
        //U!(1,1,end2%,end1%)=-U!(1,1,end1%,end1%)
        //U!(2,1,end2%,end1%)=-U!(1,2,end1%,end1%)
        //U!(3,1,end2%,end1%)=U!(3,1,end1%,end1%)
        estiff.array(4,1,-estiff.array(1,1));
        estiff.array(5,1,-estiff.array(2,1));
        estiff.array(6,1,estiff.array(3,1));
        
        // block symmeterize
        //U!(1,2,end2%,end1%)=U!(2,1,end2%,end1%)
        estiff.array(4,2,estiff.array(5,1));        
        
        //U!(2,2,end2%,end1%)=-U!(2,2,end1%,end1%)
        //U!(3,2,end2%,end1%)=U!(3,2,end1%,end1%)
        estiff.array(5,2,-estiff.array(2,2));
        estiff.array(6,2,estiff.array(3,2));
        
        //U!(1,3,end2%,end1%)=-U!(3,1,end2%,end1%)
        estiff.array(4,3,-estiff.array(6,1));
        
        //U!(2,3,end2%,end1%)=-U!(3,2,end1%,end1%)
        estiff.array(5,3,-estiff.array(3,2));
        
        //U!(3,3,end2%,end1%)=.5*U!(3,3,end1%,end1%)
        estiff.array(6,3,.5*estiff.array(3,3));
        
        // fill block 2,2
        //U!(1,1,end2%,end2%)=U!(1,1,end1%,end1%)
        //U!(2,1,end2%,end2%)=U!(2,1,end1%,end1%)
        //U!(3,1,end2%,end2%)=-U!(3,1,end1%,end1%)
        //U!(2,2,end2%,end2%)=U!(2,2,end1%,end1%)
        //U!(3,2,end2%,end2%)=-U!(3,2,end1%,end1%)
        //U!(3,3,end2%,end2%)=U!(3,3,end1%,end1%)
        estiff.array(4,4,estiff.array(1,1));
        estiff.array(5,4,estiff.array(2,1));
        estiff.array(6,4,-estiff.array(3,1));
        estiff.array(5,5,estiff.array(2,2));
        estiff.array(6,5,-estiff.array(3,2));
        estiff.array(6,6,estiff.array(3,3));
       
        // symmeterize 2,2 -- done with symm()
        //U!(1,2,end2%,end2%)=U!(1,2,end1%,end1%)                
        //U!(1,3,end2%,end2%)=-U!(1,3,end1%,end1%)
        //U!(2,3,end2%,end2%)=-U!(2,3,end1%,end1%)

	estiff.symm(); // symmeterize it

	return(estiff);
}

void truss_model::solve() // solve the problem
{
/*
	need to add singularity checks 
	
	calling function should check that there is:
	
		at least one element
		at least one restraint in X and Y directions
		at least 3 restraints
		
	if divide by zero occurs in matrix::solve, then
	
	valid_solution is not set to 1 on completion
		
*/
int elem,n1,n2,i,row,n;

	if( num_elems == 0 || valid_solution == 1)
		return;
		
	extra_rot_eqns = 0; // for pin joints in a frame
		
	Matrix stiff(3*num_nodes); // starting size of the stiffness matrix
	Matrix estiff(6); // element stiffness matrix
	
	for(elem=1;elem<= num_elems;elem++)
	{
	   n1 = node1(elem);
	   n2 = node2(elem);
	   
	   estiff=elem_stiff(elem);
	   
	   //cout << "element "<< elem<<"\n";
	   //cout << estiff;
		
	   if(model_type == TRUSS)
	   {
		stiff.assem(estiff,n1,n2,3);
	   }
	   else // its a frame
	   {
		// don't need extra equations if both nodes are rigid
		if( node_type(n1)== RIGID && node_type(n2) == RIGID)
		{
			stiff.assem(estiff,n1,n2,3);
		}
/*
	have to add equations to solve for the individual 
	end rotations of the beams that connect to pins
*/
		else 
		{
	
	/* the pinned nodes for a frame get treated here ... */
	
	
	
	
		}// if(node_type ....
		
	    } //if(mode_type== ....
	
	} // for(elem= ...	
	
	
	Matrix stiff_org(stiff); // need this later
	
	
/*
	trivial equations to set rotations to 0 for truss pins
*/	
	
	if( model_type == TRUSS)
	{
	     for(i=1;i<=num_nodes;i++)
	     {
		stiff.triv_row(3*i,0.0); // rotation at node i = 0 eqn
	     }
	}
/*
	set up the rhs -- forces
*/	
	
	// nodal and distributed forces
	if(num_loads > 0)
	{
	     for(i=1;i<=num_loads;i++)
	     {	
	     
	     	if( load_dir(i) != DISTRIBUTED )
	     	{
	     		row = (load_node(i)-1)*3+load_dir(i); 
	     		stiff.rhs_add(row,load_val(i));
	     	}
	     	else // its a distributed element force
	     	{
	     		float length,cosine,sine,delx,dely;
	     		elem = load_node(i);
        		bar_geo(&length,&cosine,&sine,&delx,&dely,elem);
        		
	     	/* ... */
	     	}
	     }
	 }
/*
	set up the restraint equations
*/	
	 
	 // specified displacements and rotations
	 if( num_restraints > 0 )
	 {
	 	for(i=1;i<=num_restraints; i++)
	 	{
	 		row = (restraint_node(i)-1)*3 + restraint_dir(i);
	 		stiff.triv_row(row,restraint_val(i));	 		 	
	 	}	 
	 }
		
	//cout << "Stiffness Matrix:\n";
	//cout << stiff;
	
	Vector ans(3*num_nodes+extra_rot_eqns);
	int ret; // the return code from the solver
	
	ans=stiff.solve(&ret);
	
	if( ret != 0 ) // not OK
	{
		return;
	}
	
	//cout << "solution:\n";
	//cout << ans;
	
/*
	save the nodal displacements
*/	
	
	node_us = (float *)malloc(num_nodes*sizeof(float));
	if( node_us == 0)
		die();
	node_vs = (float *)malloc(num_nodes*sizeof(float));
	if( node_vs == 0)
		die();
	node_rots = (float *)malloc(num_nodes*sizeof(float));
	if( node_rots == 0)
		die();
		
	n=0;		
	for(i=1;i<=3*num_nodes;i+=3)
	{
		node_us[n]=ans.val(i);
		node_vs[n]=ans.val(i+1);
		node_rots[n]=ans.val(i+2);
		n ++;
	}
		
	//cout << "checking:\n";
	
	//Vector chk(3*num_nodes+extra_rot_eqns);
	//chk= stiff_org*ans;
	//cout << chk;
	
/* 
	calculate and save reaction forces for the restraints
	
	needs mods for distributed forces
*/		
	Vector force(3*num_nodes+extra_rot_eqns);
	force=stiff_org*ans;
	
	restraint_forces=(float *)malloc(num_restraints*sizeof(float));
	if(restraint_forces == 0)
		die();
	
	for(i=1;i<=num_restraints;i++)
	{
		row = (restraint_node(i)-1)*3 + restraint_dir(i);
		restraint_forces[i-1]=force.val(row);
	}
	
	//cout << "force:\n";
	//cout << force;

/*
	calculate the bar forces		
*/
	Vector elem_disp(6); // displacements in local indices
	Vector elem_force(6); // element forces in local indices
	float length,cosine,sine,delx,dely;
	
	elem_axial = (float *)malloc(num_elems*sizeof(float));
	if(elem_axial == 0)
		die();
	
	for(elem=1;elem<= num_elems;elem++)
	{
	   n1 = node1(elem);
	   for(i=1;i<=3;i++)
	   {
	   	elem_disp.val(i,ans.val(3*(n1-1)+i));
	   }
	   n2 = node2(elem);
	   for(i=1;i<=3;i++)
	   {
	   	elem_disp.val(i+3,ans.val(3*(n2-1)+i));
	   }
	   
	   estiff=elem_stiff(elem);
	   
	   elem_force=estiff*elem_disp;
	   
	   bar_geo(&length,&cosine,&sine,&delx,&dely,elem);

	   // -1 make it positive in tension
	   elem_axial[elem-1]=-1.*(cosine*elem_force.val(1)+
	   		      sine*elem_force.val(2));
	   		      
	   /* for a frame need shear and moment too .... */
	   
	}
		
	valid_solution = 1;
	
}

float truss_model::max_disp()
{
int i;
float u,v,d,dmax;

	if(valid_solution != 1)
		return(1.0);
		
	dmax = 0.0;
	
	for(i=0;i<num_nodes;i++)
	{
		u=node_us[i];
		v=node_vs[i];
		d=sqrt(u*u+v*v);
		
		if(d > dmax)
			dmax=d;
	}
	
	return(dmax);
}

FPoint truss_model::disp(int i)
{
/*
	returns u,v of node number i
	
*/
FPoint U;
	if( i < 1 || i > num_nodes || valid_solution == 0)
		return(U);

	U.setX(node_us[i-1]);
	U.setY(node_vs[i-1]);
	
	return(U);
}

float truss_model::rot(int i)
{
/*
	returns the rotation of node i
*/
	if( i < 1 || i > num_nodes || valid_solution == 0)
		return(0.);
		
	return(node_rots[i-1]);
}

FPoint truss_model::disp_node(int i,float scale)
{
/*
	returns the displaced node position 
	
	for node number i
	
	and displacements are scaled by scale.
	
*/	
FPoint p,U,n;
	if( i < 1 || i > num_nodes)
		return(p); // zero point

	U=disp(i);
	
	n=node(i);
	
	p=n+scale*U;

	return(p);
}

int truss_model::solved() const
{
	return(valid_solution);
}

void truss_model::discard_solution()
{
/*
	model change invalidates solution
*/
	if(valid_solution == 1)
	{
		free(node_us);
		free(node_vs);
		free(node_rots);
		
		free(restraint_forces);	
		
		free(elem_axial);
		
		free(rot_eqns_elem);
		free(rot_eqns_node);
	
	
		valid_solution = 0; 
	}
}

void truss_model::save_rot_eqn(int elem,int node)
{

	if( elem < 1 || elem > num_elems)
	{
		cout << " range error on elem in save_rot_eqn\n";
		return;
	}
	if( node < 1 || node > 2)
	{
		cout << " range error on node in save_rot_eqn\n";
		return;
	}
	extra_rot_eqns++;
	
	rot_eqns_elem = (int *)realloc(rot_eqns_elem,extra_rot_eqns*sizeof(int));
	if(rot_eqns_elem == 0)
		die();
	
	rot_eqns_node = (int *)realloc(rot_eqns_node,extra_rot_eqns*sizeof(int));
	if(rot_eqns_node == 0)
		die();
	
	rot_eqns_elem[extra_rot_eqns-1] = elem;
	rot_eqns_node[extra_rot_eqns-1] = node;
}	
