/*
	
    This code is part of Truss Solver which is free software.
    See the file COPYING for more details.

    Copyright (C) 1997  T. W. Shield, shield@aem.umn.edu

	class truss_model

	This class provides the data structures and calculations
	for truss (and eventually frame) 2-D structural models.

	This class provides support for adding, deleting, and modifing
	
		nodes
		elements
		material property sets (mpsets)
		applied loads
		applied restraints

	as well as file i/o (simple ascii file format)
	
	all memory is allocated dynamically via malloc/free
	
	thus problem size is only limited by available memory, but
	no attempt has been made to program for speed.

	requires the following classes:
	
		Vector (included by matrix)
		Matrix
		FPoint
		
	the linear equation solver is in the matrix class.
	
	Indexes in all functions start at 1 (one).
	
	WARNING:
	This class does not have a copy constructor or an operator=
	defined.  Thus, you cannot manipulate multiple models with it.
	
	This class can be used independantly from the Point and Click 
	interface written in Qt.  Simply change the define to undef in 
	the line below and compile with the required classes listed 
	above.

	change the following to undef if you are not using QT
	
*/	
#define QT

#ifdef QT
    #include <qtstream.h>
    #define ISTREAM QTextStream
    #define OSTREAM QTextStream
#else
    #include <iostream.h>
    #define ISTREAM istream
    #define OSTREAM ostream
#endif

#include "fpoint.h"
#include "matrix.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

// return from adding elements
#define OK 0
#define BAD 1
#define DUP 2

// return from reading load and restraint files (along with OK and BAD)
#define BAD_NODE 3
#define BAD_ELEM 4

// for the BC's, note values are NOT arbitrary!!!
#define X_DIRECTION 1
#define Y_DIRECTION 2
#define ROTATION 3
#define MOMENT 3
#define DISTRIBUTED 4

// for the node type
#define PIN 0
#define RIGID 1

// for model type
#define TRUSS 0
#define FRAME 1

// for the coincident node check -- as a fraction of (xmax-xmin)
const float NODE_CHK_TOL=0.05;

class truss_model
{

public:
	truss_model();
	~truss_model();

/* nodes */
	int nodes(); // number of nodes
	void add_node(float,float);
	void add_node(FPoint);
	void set_node_pos(int,FPoint);
	void set_node_pos(int,float,float);
	void set_node_type(int,int);
	void delete_node(int);	
	float node_x(int);
	float node_y(int);
	FPoint node(int);
	int node_type(int); // returns the node type
	int connects(int); // number of connections to a node
	int node_in_bc(int);// is the node used in bc's? 0=no
	int coincident_node_chk(int n,FPoint p); // returns 0 if ok
			// returns to close node number otherwise

/* elements */	
	int elems(); // number of elements
	int add_elem(int,int,int); // returns OK or BAD/DUP on bad request
	int update_elem(int,int,int,int);
	void delete_elem(int);
	float elem_x1(int);
	float elem_y1(int);
	FPoint elem_n1(int); // location of node 1 on elem	
	float elem_x2(int);
	float elem_y2(int);
	FPoint elem_n2(int); // location of node 2 on elem
	int mpset(int);
	int node1(int); // node 1 number on elem
	int node2(int); // node 2 number on elem
	FPoint elem_center(int); // center point of the element
	int elem_in_bc(int);// is the elem used in bc's? 0=no
	
/* material property (MP) sets */	
	int mpsets(); // number of mp sets
	void add_mpset_rod(float,float);
	void update_mpset_rod(int,float,float);
	void update_mpset_beam(int,float,float,float,float,float);	
	void delete_mpset(int);
	float area(int); // area of an mpset
	float modulus(int); // modulus of an mpset
	float I(int); // moment of inertia of an mpset
	float Q(int); // first moment Q
	float SR1(int); //stress recovery point 1
	float SR2(int); //stress recovery point 2
	float NA_width(int);//width at Neutral axis
	int mpset_used(int); // number of times mpset is used

	
/* whole model */
	float xmin();
	float xmax();
	float ymin();
	float ymax();
	int empty(); // returns 0 if the model is empty
	int type();
	void set_type(int); // TRUSS or FRAME, model must be empty
				
/* boundary conditions */

	int add_load(int node,int direction,float value); // returns OK or DUP
	int add_restraint(int,int,float); // returns OK or DUP
	
	int loads(); // returns number of loads
	int restraints(); // returns number of restraints
	
	void delete_load(int,int); // args are node,direction
	void delete_restraint(int,int);
	
	int load_node(int); // returns the node/elem for load i
	int load_dir(int);  //returns the direction
	float load_val(int);

	int restraint_node(int); // returns the node/elem for load i
	int restraint_dir(int);  //returns the direction
	float restraint_val(int);
	
	void load_dir_str(int,char *); // direction text
	void restraint_dir_str(int,char *); // direction text
	
// solving the problems

	void bar_geo(float *,float *,float *,float *,float *,int ); 
	Matrix elem_stiff(int);	
	void solve();
	float max_disp(); // returns the maximum displacement magnitude
	int solved() const; // returns valid_solution = 1 have solution
	FPoint disp_node(int,float);// returns displaced node position
			// for int node with displacements scaled by float
	FPoint disp(int); // returns the displacement of a node
	float rot(int); // returns the rotation of the node
	
/* misc */	

	void example();

	void print(OSTREAM *); // does all of the below
	void print_model(OSTREAM *);
	void print_loads(OSTREAM *);
	void print_restraints(OSTREAM *);
	void print_results(OSTREAM *);
	
	int input_all(ISTREAM *); // returns OK or BAD
	int input_loads(ISTREAM *); // returns OK or BAD
	int input_restraints(ISTREAM *); // returns OK or BAD

private:

/* make sure this are not used */
	truss_model(truss_model &); // copy constructor
	void operator=(truss_model ); // = operator

	/* for nodes */

	int num_nodes;

	float *node_xs,*node_ys; // node x and y positions
	float *node_us,*node_vs,*node_rots;// nodal displacements (u,v,theta)
	int *connections; // count of connections to the node
	int *node_types; // RIGID or PIN
	int *node_bcs;// count of loads/restraints on the node
	
	/* for elements */

	int num_elems;

	int *elem_node1,*elem_node2,*elem_mpset;
	int *elem_bcs; // count of distributed loads on the element
	float *elem_axial; // axial force in the element	

	int check_duplicate(int,int,int); // check for duplicate elements

	/* for mpsets */
	
	int num_mpsets;
	
	float *mpset_area,*mpset_E,*mpset_I;
	float *mpset_Q,*mpset_SR1,*mpset_SR2,*mpset_NA_width;
	int *mpset_uses; // array counting how many elements use an mpset
	
	/* for the BC */

	int num_loads;
	int *load_nodes; // is an element number if DISTRIBUTED
	int *load_dirs; // X_DIRECTION, Y_DIRECTION or DISTRIBUTED
	float *load_vals;
	
	int num_restraints;
	int *restraint_nodes;
	int *restraint_dirs;//  X_DIRECTION, Y_DIRECTION or ROTATION
	float *restraint_vals;
	
	int check_dup_bc(int,int);
	
	int model_type; // TRUSS or FRAME

	/* for solving */
		
	int valid_solution; // 1 if we have a solution
	void discard_solution(); // call to invalidate solution
				// frees the solution allocations if needed
	
	int extra_rot_eqns; // number of extra rotation equations needed
	
	int *rot_eqns_elem; //element which the rot eqn is form
	int *rot_eqns_node; // 1 or 2 which node on the element
	
	void save_rot_eqn(int,int);
	
	float *restraint_forces;// index is number of the restraint

	/*  for memory allocation, also do free's
	    if the num_ variable is zero	
	*/
	
	void alloc_nodes();
	void alloc_elems();
	void alloc_mpsets();
	void alloc_loads();
	void alloc_restraints();
	
	void die(); // for alloc errors
	
	// for i/o
	
	int read_model(ISTREAM *); // these return OK or BAD
	int read_loads(ISTREAM *);
	int read_restraints(ISTREAM *);
	
	void write_model(OSTREAM *);
	
	void set_counters();// calculate connections, mpset_uses,
			   // node_bcs and elem_bcs
	void set_bc_counters(); // called by set_counters, just does
			// restraint and load counters
			   
	int parse_i(char *,char *,int *); // parsing for input_all
	
	void getline(ISTREAM *,char *,int); 
		
};

OSTREAM &operator<<(OSTREAM &os, truss_model &truss);


