/*

    This code is part of Truss Solver which is free software.
    See the file COPYING for more details.

    Copyright (C) 1997  T. W. Shield, shield@aem.umn.edu

*/

#include <qapp.h>
#include <qfont.h>
#include <qpoint.h>
#include <qrect.h>
#include <qfiledlg.h> 
#include <qstring.h>
#include <qtstream.h>
#include <qfile.h>
#include <qkeycode.h>
#include <qprinter.h>
#include <qdatetm.h>
#include <qcursor.h>
#include <iostream.h>
#include <fstream.h>
#include <math.h>
#include <stdio.h>
#include "global.h"
#include "yesnoinfo.h"
#include "bcs/Edit_Bcs.h"
#include "viewtext/ViewText.h"
#include "icons/main_icon.xpm"
#include "version.h"


mywin::mywin()  // default constructor
{

	init();
	
	node_size.setWidth(NODE_SIZE);
	node_size.setHeight(NODE_SIZE);
	
	setMinimumSize( 200,200);
	
			
/*
	menu declarations
	
*/	
    saves_menu = new QPopupMenu();
    CHECK_PTR( saves_menu );
    save_model_id = saves_menu->insertItem("Model",this, SLOT(save_model()));
    save_loads_id = saves_menu->insertItem("Loads",this, SLOT(save_loads()));
    save_restraints_id = saves_menu->insertItem("Restraints",this, SLOT(save_restraints()));
    save_all_id = saves_menu->insertItem("Results",this, SLOT(save_all()));
 
// these are changed in repaint as needed    
    saves_menu->setItemEnabled(save_model_id, FALSE);
    saves_menu->setItemEnabled(save_loads_id, FALSE);
    saves_menu->setItemEnabled(save_restraints_id, FALSE);
    saves_menu->setItemEnabled(save_all_id, FALSE);
    
    opens_menu = new QPopupMenu();
    CHECK_PTR( opens_menu );
    opens_menu->insertItem("Model",this, SLOT(input_model()));
    open_loads_id = opens_menu->insertItem("Loads",this, SLOT(input_loads()));
    open_restraints_id = opens_menu->insertItem("Restraints",this, SLOT(input_restraints()));
 	
// these are changed in repaint as needed    
    opens_menu->setItemEnabled(open_loads_id, FALSE);
    opens_menu->setItemEnabled(open_restraints_id, FALSE);
	
    file_menu = new QPopupMenu();
    CHECK_PTR( file_menu );
    file_menu->insertItem("New", this, SLOT(new_model()) );
    file_menu->insertItem("Open",opens_menu);
    file_menu->insertItem("Example", this, SLOT(example_model()) );
    file_menu->insertSeparator();
    file_menu->insertItem("Save", saves_menu);
    file_menu->insertSeparator();
    file_menu->insertItem("Print", this, SLOT(print()) );
    file_menu->insertSeparator();
    file_menu->insertItem( "Exit", this, SLOT(exit()), CTRL+Key_Q );
    
    model_menu = new QPopupMenu();
    CHECK_PTR( model_menu );
#ifdef ALLOW_FRAME    
    model_type_id = model_menu -> insertItem("Truss",this, SLOT(model_type()));
    model_menu -> insertSeparator();
#endif
    model_menu -> insertItem("Nodes...", this, SLOT(node_edit()) ); 
    model_menu -> insertItem("MP sets...", this, SLOT(mpset_edit()) );       
    model_menu -> insertItem("Elements...", this, SLOT(element_edit()) );       
    model_menu -> insertItem("Loads/Restraints...", this, SLOT(bc_edit()) );       
    model_menu -> insertSeparator();
    solve_model_id = model_menu -> insertItem("Solve", this, SLOT(solve())  , CTRL+Key_S); 
    view_solution_id = model_menu -> insertItem("View Results",this,SLOT(view_solution()));
    
    model_menu->setItemEnabled(solve_model_id, FALSE);
    model_menu->setItemEnabled(view_solution_id, FALSE);
    
    options_menu = new QPopupMenu();
    
    show_nodes = 1;
    show_nodes_id = options_menu->insertItem("Hide nodes",this,SLOT(toggle_show_nodes()));
    options_menu->insertItem("Snap...",this,SLOT(set_snap()));

    //options_menu->setItemChecked(show_nodes_id, TRUE);

    help_menu = new QPopupMenu();
    help_menu-> insertItem("Help...",this, SLOT(help()));
    help_menu-> insertItem("About", this, SLOT(about()));
    
    menu = new QMenuBar( this );
    CHECK_PTR( menu );
    menu->insertItem( "File", file_menu );
    menu->insertItem( "Model", model_menu);
    menu->insertItem("Options",options_menu);
    menu->insertSeparator();
    menu->insertItem("Help",help_menu);
    
/*
	set the window icon and text and window caption
*/


    QPixmap win_icon(main_icon);
    
    setIcon(win_icon);

    const char title[]="Truss Solver";

    setCaption(title);
    
    setIconText(title);
	
}

void mywin::init() // stuff we need to do on new as well as startup
{
	xmax = 1.0;
	xmin = 0.;
	ymax = 1.0;
	ymin = 0.;
	
	snap_spacing = 0.; // no snap/grid
		
	setmatrix();
	
}

void mywin::setmatrix()  // sets the coordinate transformation variables
{
        scalex = (width())/(xmax-xmin);
        scaley = (height())/(ymax-ymin);


	if(scalex > scaley ) // keep aspect square
		scalex = scaley;
	else
		scaley = scalex;


	scaley = -scaley;

        transx = -xmin*scalex;
        transy = -ymax*scaley;
}

QPoint mywin::coords(float x,float y)  // returns a screen position point from
				      // model position (x,y)
{
	QPoint p;
	
	p.setX((int)(scalex*x + transx));
	p.setY((int)(scaley*y + transy));


	return(p);
}

QPoint mywin::coords(FPoint fp)  // returns a screen position point from
				      // model position (x,y)
{
	QPoint p;
	
	p.setX((int)(scalex*fp.x() + transx));
	p.setY((int)(scaley*fp.y() + transy));


	return(p);
}

FPoint mywin::icoords(QPoint p) // inverse coord transformation
{
	FPoint fp;
	
	fp.setX((((float)p.x())-transx)/scalex);
	fp.setY((((float)p.y())-transy)/scaley);
	
	return(fp);
}

void mywin::set_limits()
/*
	sets the window edge values in model coords
*/	
{
float bx,by,b,topb;
float xmaxt,xmint,ymaxt,ymint;

	xmaxt = truss->xmax();
	xmint = truss->xmin();
	ymaxt = truss->ymax();
	ymint = truss->ymin();
	
	if(xmaxt == xmint) xmaxt = xmaxt + 1.;
	if(ymaxt == ymint) ymaxt = ymaxt + 1.;
	
	topb = (menu->height()+NODE_SIZE)*(ymaxt-ymint)/(height()-menu->height()-NODE_SIZE); // size of menubar at top in model coords

	bx = .1*(xmaxt-xmint); // add a border 
	by = .1*(ymaxt-ymint);
	if(bx > by )
		b=bx;
	else
		b=by;
	b=fabs(b);
	xmax = xmaxt+b;
	xmin = xmint-b;
	ymax = ymaxt+b+topb;
	ymin = ymint-b;

	setmatrix();
}

void mywin::paintEvent(QPaintEvent *)  // draws the truss on the screen
{
	
	if( truss->empty() != 0) // only paint non-empty model
	{
		draw(this);
		
	}     
/*

	set save menu enables -- paint is called every time something changes

*/
	if(truss->empty() == 0)
	{
    		opens_menu->setItemEnabled(open_loads_id, FALSE);
    		opens_menu->setItemEnabled(open_restraints_id, FALSE);
		saves_menu->setItemEnabled(save_model_id, FALSE);
	}
	else
	{
    		opens_menu->setItemEnabled(open_loads_id, TRUE);
    		opens_menu->setItemEnabled(open_restraints_id, TRUE);
		saves_menu->setItemEnabled(save_model_id, TRUE);
	}
		
	if( truss->elems() == 0)
	{
		model_menu->setItemEnabled(solve_model_id, FALSE);
	}
	else
	{
		model_menu->setItemEnabled(solve_model_id, TRUE);
	}
	
	if(truss->loads() == 0)
    		saves_menu->setItemEnabled(save_loads_id, FALSE);
    	else
    		saves_menu->setItemEnabled(save_loads_id, TRUE);
    	
	if(truss->restraints() == 0)
    		saves_menu->setItemEnabled(save_restraints_id, FALSE);
    	else
    		saves_menu->setItemEnabled(save_restraints_id, TRUE);
    	
	if(truss->solved() == 0)
	{
    		saves_menu->setItemEnabled(save_all_id, FALSE);
		model_menu->setItemEnabled(view_solution_id, FALSE);
    	}
 	else
 	{       
    		saves_menu->setItemEnabled(save_all_id, TRUE);
		model_menu->setItemEnabled(view_solution_id,TRUE);
    	}
}

void mywin::draw(QPaintDevice *w, int prt_flag)
{
/*

	prt_flag = 0 (the default) is for the screen
	
		 = 1 is for printing to portrait
		 
		 = 2 is for printing to landscape

*/		 
	float x1,x2,y1,y2,dscale;
	FPoint p1,p2;
	int i;

	dscale = 1.0;
	
	QPainter paint(w);

/*

	are we out putting to the printer?
	
	if so have to set up the scaling and font
	defaults are not platform independant
	
*/	
	if(prt_flag == 1) 
	{
		QFont f( "times", 12);
    		paint.setFont( f );
		paint.setWindow(0,0,612,828); // portrait coords
	}
	if(prt_flag == 2)
	{
		QFont f( "times", 12);
    		paint.setFont( f );
		paint.setWindow(0,0,828,612); // landscape coords
	}
/*

	snap grids -- first so they are under everything
	
*/	
	if( prt_flag == 0 && snap_spacing > 0.)  // don't print these
	{
		QPen grid_pen(gray,1,DashLine);
		
		paint.setPen(grid_pen);
		
		FPoint up_left,low_right;
		
		QPoint q00(0,0);
		
		up_left=icoords(q00);
		
		QPoint qwh(width(),height());
		
		low_right=icoords(qwh);
		
		float x,y;
		
		x = ((int)(up_left.x()/snap_spacing))*snap_spacing; // start value
		
		while( x < low_right.x() )
		{
			paint.drawLine(coords(x,low_right.y()),coords(x,up_left.y()));
			
			x += snap_spacing;
		}
	
		y = ((int)(low_right.y()/snap_spacing))*snap_spacing; // start value
		
		while( y < up_left.y() )
		{
			paint.drawLine(coords(up_left.x(),y),coords(low_right.x(),y));
			
			y += snap_spacing;
		}			
	
	}

// some needed pens -- must create with thickness 1 for printing

        QPen black_pen(black,1);
        QPen green_pen(green,1);
        QPen darkGray_pen(darkGray,1);
        
	paint.setPen( black_pen );
	
/*
	draw lines for the elements
*/

	for(i=1;i<= truss->elems();i++)
	{
		if( truss->solved() == 0)
		{
		    if(truss->elem_in_bc(i) == 0)
		    {
			paint.setPen( black_pen );
		    }
		    else
		    {
			paint.setPen( green_pen ); // elems with distributed forces
						// draw green
		    }
		}
		else
		{
		 	paint.setPen( darkGray_pen ); // original shape when solved
		 				// draws darkGray
		}
			
		x1=truss->elem_x1(i);
		y1=truss->elem_y1(i);
		x2=truss->elem_x2(i);
		y2=truss->elem_y2(i);
		
		paint.drawLine(coords(x1,y1),coords(x2,y2));
	}
/*
	draw lines for the displaced elements
	
	if problem is solved
	
	needs frame mods
*/
	if( truss->solved() == 1)
	{
		paint.setPen( black_pen );
	
		dscale=0.1*(truss->xmax() - truss->xmin())/truss->max_disp();
		
		for(i=1;i<= truss->elems();i++)
		{					
			p1=truss->disp_node(truss->node1(i),dscale);
			p2=truss->disp_node(truss->node2(i),dscale);
		
			paint.drawLine(coords(p1),coords(p2));
		}
	}

/*
	draw the nodes with number text
*/
	
	char temp[50];
	QPoint p;
	QRect r;
	QBrush brush;
	
	brush.setColor(black);
	brush.setStyle(SolidPattern);
	paint.setPen(white); // node number text is white
	
	r.setSize(node_size);
	
	
	if(show_nodes == 1)
	{
	    for(i=1;i<=truss->nodes();i++)
	    {
		if(truss->connects(i) == 0)
		{
			brush.setColor(darkGray); // unconnected nodes are gray
		}
		else
		{
			brush.setColor(black);
		}
		
		if(truss->node_in_bc(i) != 0)
		{
			brush.setColor(blue); // nodes with BC are blue
		}

			
		sprintf(temp,"%i",i);
		
		x1=truss->node_x(i);
		y1=truss->node_y(i);
		p=coords(x1,y1);
		
		r.moveCenter(p);
		
		paint.fillRect(r,brush);
		paint.drawText(r,AlignCenter,temp,strlen(temp));		

	   }
	}
	
/*
	text, displacement mag and time/date on printouts
	
*/	
	
	if( truss->solved() == 1) // show the displacement magnification factor
	{	
		paint.setPen(black); 
		sprintf(temp,"Displacement Mag = %g", dscale);
		paint.drawText(0,40,temp,strlen(temp));
	}
	
	if( prt_flag != 0) // we are printing, add date and time
	{
		QDateTime dt = QDateTime::currentDateTime();
		QString qs;
			qs = dt.toString();
			strcpy(temp,qs);
		paint.drawText(300,40,temp,strlen(temp));
	}

        paint.end();                            // painting done
        
}

void mywin::resizeEvent(QResizeEvent *)
{
	setmatrix();    	
}

void mywin::mouseReleaseEvent(QMouseEvent *me)
/*
	if a node is clicked, returns the node number with
	
		signal node_clicked(int node)
		
	if element center is clicked the returns the element number with
	
		signal elem_clicked(int elem)
		
	else emits a
	
		signal clicked(FPoint pos)
		
		with the screen position in model coords
		
		
*/		
{

	QPoint node;
	FPoint pos;
	int i;
	
	pos=icoords(me->pos()); // model position of the click point
	
	if(truss->nodes() > 0)
	{

		for(i=1;i<=truss->nodes();i++)
		{
			node=coords(truss->node_x(i),truss->node_y(i));	
		
			if( qpdist(me->pos(),node) < node_size.width()/2 )
			{
				emit node_clicked(i);
				return;
			}
		}
		
		for(i=1;i<=truss->elems();i++)
		{
			node=coords(truss->elem_center(i));	
		
			if( qpdist(me->pos(),node) < node_size.width()/2 )
			{
				emit elem_clicked(i);
				return;
			}
		}

	}

	if( snap_spacing > 0.)
	{
		pos.setX(((int)(pos.x()/snap_spacing+.5))*snap_spacing);
		pos.setY(((int)(pos.y()/snap_spacing+.5))*snap_spacing);
	}


	emit clicked(pos);
}

/*******************************************************************

	slots
	
********************************************************************/	

void mywin::show_node(int node)
{
	cout << "Node clicked is "<< node << "\n";
		
}
	
void mywin::show_position(FPoint p)
{
	cout << "position clicked is "<< p << "\n";
}	

void mywin::print()  // file menu print item
{
QPrinter prt;
int orient;

	prt.setPrintProgram("lpr");
	prt.setPageSize(QPrinter::Letter);

        if ( prt.setup(0) ) 
        {
        
// set up scaling for the printer
        		
        	if( prt.orientation() == QPrinter::Portrait )
        	{
			scalex = (72.*8.5)/(xmax-xmin); // units are points, width
        		scaley = (72.*11.)/(ymax-ymin);// height
        		
        		orient = 1;
		}
		else
		{		
			scalex = (72.*11.)/(xmax-xmin); // units are points, width
        		scaley = (72.*8.5)/(ymax-ymin);// height
        		
        		orient = 2;
		}

		if(scalex > scaley ) // keep aspect square
			scalex = scaley;
		else
			scaley = scalex;


		scaley = -scaley;

        	transx = -xmin*scalex;
        	transy = -ymax*scaley;
        	       	
// draw it 
       	
        	draw(&prt,orient);
        
// reset to screen scaling

        	setmatrix(); 
        }

}

void mywin::exit()	//file menu exit item
{
	if(truss->empty() != 0)
	{
		YesNoDialog yesno(this,"Quit?");
		if(yesno.exec())
			emit quit();
	}
	else
		emit quit();
}

void mywin::node_edit() // add/del nodes
{

	if(!node_win->isVisible())
	{
		node_win->init();
		
		node_win->show();
	}
		
		
}

void mywin::element_edit() // add/del elements
{
	if(!elem_win->isVisible())
	{
		elem_win->init();
	
		elem_win->show();
	}
}

void mywin::mpset_edit() // add/del elements
{
	if(!mpset_win->isVisible())
	{
		mpset_win->init();
	
		mpset_win->show();
	}
}

void mywin::model_type() // toggle frame/truss
{
	if(truss->empty() == 0)
	{
		if(truss->type() == FRAME)
		{
			model_menu->changeItem("Truss",model_type_id);
			truss->set_type(TRUSS);
		}
		else
		{
			model_menu->changeItem("Frame",model_type_id);
			truss->set_type(FRAME);
		}
		
		if(bcs_win->isVisible()) // take care of button changes
			bcs_win->init();
			
		if(node_win->isVisible())
			node_win->init();
	}
	else
	{
		InfoDialog info(this,"Can only change model type for new model");
		info.exec();
	}
}

void mywin::bc_edit() // add/del loads and restraints
{
	if(!bcs_win->isVisible())
	{
		bcs_win->init();
		
		bcs_win->show();
	}
}

void mywin::new_model() // start with clean slate
{
	
	if( truss->empty() == 1) // not empty
	{	
		YesNoDialog yesno(this,"Discard current model ?");
		
		if(yesno.exec())
		{
			delete truss;
			truss = new truss_model;
			
			if(show_nodes == 0) // set show_nodes to yes
				toggle_show_nodes();
				
			init(); // reset the scaling
			
			repaint();
		}
	
	}
}

void mywin::example_model() // set up the truss example
{
	
	if( truss->empty() == 1) // not empty
	{
	     YesNoDialog yesno(this,"Discard current model ?");
	     
	     if(yesno.exec())
	     {
		delete truss;
		truss = new truss_model;

	     }
	     else
	     	return;
	}

	if(show_nodes == 0) // set show_nodes to yes
		toggle_show_nodes();
				
	init(); // reset the scaling
	
	truss->example();
	set_limits();
	repaint();
}

void mywin::solve()
{
int n_rts,i,x_dir,y_dir;

	n_rts = truss->restraints();

	if( (n_rts+truss->loads()) < 4)
	{
		InfoDialog info(this,"Not enough Loads and/or Restraints to solve");
		info.exec();
		return;
	}
	
	if( n_rts < 3)
	{
		InfoDialog info(this,"Not enough Restraints to solve");
		info.exec();
		return;
	}
	
	x_dir = 0;
	y_dir = 0;
	
	for(i=1;i<=n_rts;i++)
	{
		if(truss->restraint_dir(i) == X_DIRECTION)
			x_dir++;
		if(truss->restraint_dir(i) == Y_DIRECTION)
			y_dir++;
	}
	
	if( x_dir < 1)
	{
		InfoDialog info(this,"Must have at least one X-Direction Restraint");
		info.exec();
		return;
	}
	
	if( y_dir < 1)
	{
		InfoDialog info(this,"Must have at least one Y-Direction Restraint");
		info.exec();
		return;
	}
	
	
	QCursor org_cursor(cursor());
	
	QApplication::setOverrideCursor(waitCursor);
	
	truss->solve();
	
	QApplication::setOverrideCursor(org_cursor);
	
	if(truss->solved() != 1)
	{
		InfoDialog info(this,"Error Occured during Solve, Check your Model!");
		info.exec();
	}	
	repaint(); // to show the deformed shape

}

void mywin::save_all()
{
unsigned int pos;

char filter[10];

	strcpy(filter,"*");
	strcat(filter,RESULTS_EXT);

	QString f = QFileDialog::getSaveFileName( previous_dir, filter, this );

	if ( !f.isEmpty() ) // the user gave a file name
	{
		if( f.find(RESULTS_EXT) == -1 ) // force the extension
		{
			f.append(RESULTS_EXT);
		}
       		
    		pos = f.findRev('/',f.length()-1);
    			
    		if(pos > 0) // save it for later opens
    		{
    			previous_dir = f.left(pos+1);
    		}
    		
    		QFile qf(f); 
    		
    		if( qf.exists() ) // then the file exists
    		{
    			YesNoDialog yesno(this,"File exists, Overwrite ?");
    			
    			if(! yesno.exec() ) // if answer is no, return
    			{
    				return;
    			}
    		}	
    		    		
    		if( qf.open(IO_WriteOnly | IO_Translate) == FALSE )
    		{
    			InfoDialog info(this,"Open failed on file save");
    			info.exec();
    			return;
    		}

		QTextStream qs(&qf);
        
        	truss->print(&qs);
        	
        	qf.close();
        	
        	InfoDialog info(this,"Results Saved");
        	info.exec();
        
	}


}

void mywin::view_solution()
{
	QString solution_text;
	
	QTextStream qs(solution_text,IO_WriteOnly);
	
	truss->print(&qs);

	ViewText vt(solution_text);
	
	vt.exec();
}

void mywin::save_model()
{
unsigned int pos;

char filter[10];

	strcpy(filter,"*");
	strcat(filter,MODEL_EXT);

	QString f = QFileDialog::getSaveFileName( previous_dir, filter, this );

	if ( !f.isEmpty() ) // the user gave a file name
	{
		if( f.find(MODEL_EXT) == -1 ) // force the extension
		{
			f.append(MODEL_EXT);
		}
       		
    		pos = f.findRev('/',f.length()-1);
    			
    		if(pos > 0) // save it for later opens
    		{
    			previous_dir = f.left(pos+1);
    		}
    		
    		QFile qf(f); 
    		
    		if( qf.exists() ) // then the file exists
    		{
    			YesNoDialog yesno(this,"File exists, Overwrite ?");
    			
    			if(! yesno.exec() ) // if answer is no, return
    			{
    				return;
    			}
    		}	
    		    		
    		if( qf.open(IO_WriteOnly | IO_Translate) == FALSE )
    		{
    			InfoDialog info(this,"Open failed on file save");
    			info.exec();
    			return;
    		}

		QTextStream qs(&qf);
        
        	truss->print_model(&qs);
        	
        	qf.close();
        	
        	InfoDialog info(this,"Model Saved");
        	info.exec();
        
	}


}

void mywin::save_loads()
{
unsigned int pos;

char filter[10];

	strcpy(filter,"*");
	strcat(filter,LOADS_EXT);

	QString f = QFileDialog::getSaveFileName( previous_dir, filter, this );

	if ( !f.isEmpty() ) // the user gave a file name
	{
		if( f.find(LOADS_EXT) == -1 ) // force the extension
		{
			f.append(LOADS_EXT);
		}
       		
    		pos = f.findRev('/',f.length()-1);
    			
    		if(pos > 0) // save it for later opens
    		{
    			previous_dir = f.left(pos+1);
    		}
    		
    		QFile qf(f); 
    		
    		if( qf.exists() ) // then the file exists
    		{
    			YesNoDialog yesno(this,"File exists, Overwrite ?");
    			
    			if(! yesno.exec() ) // if answer is no, return
    			{
    				return;
    			}
    		}	
    		    		
    		if( qf.open(IO_WriteOnly | IO_Translate) == FALSE )
    		{
    			InfoDialog info(this,"Open failed on file save");
    			info.exec();
    			return;
    		}

		QTextStream qs(&qf);
        
        	truss->print_loads(&qs);
        	
        	qf.close();
        	
        	InfoDialog info(this,"Loads Saved");
        	info.exec();
        
	}


}

void mywin::save_restraints()
{
unsigned int pos;

char filter[10];

	strcpy(filter,"*");
	strcat(filter,RESTRAINTS_EXT);

	QString f = QFileDialog::getSaveFileName( previous_dir, filter, this );

	if ( !f.isEmpty() ) // the user gave a file name
	{
		if( f.find(RESTRAINTS_EXT) == -1 ) // force the extension
		{
			f.append(RESTRAINTS_EXT);
		}
       		
    		pos = f.findRev('/',f.length()-1);
    		
    		if(pos > 0) // save it for later opens
    		{
    			previous_dir = f.left(pos+1);
    		}
    		
    		QFile qf(f); 
    		
    		if( qf.exists() ) // then the file exists
    		{
    			YesNoDialog yesno(this,"File exists, Overwrite ?");
    			
    			if(! yesno.exec() ) // if answer is no, return
    			{
    				return;
    			}
    		}	
    		    		
    		if( qf.open(IO_WriteOnly | IO_Translate) == FALSE )
    		{
    			InfoDialog info(this,"Open failed on file save");
    			info.exec();
    			return;
    		}

		QTextStream qs(&qf);
        
        	truss->print_restraints(&qs);
        	
        	qf.close();
        	
        	InfoDialog info(this,"Restraints Saved");
        	info.exec();
        
	}


}


void mywin::input_model()
{
unsigned int pos;

char filter[10];

	strcpy(filter,"*");
	strcat(filter,MODEL_EXT);

	if( truss->empty() == 1) // not empty
	{	
		YesNoDialog yesno(this,"Discard current model ?");
		
		if( yesno.exec())
		{
			delete truss;
			truss = new truss_model;
		}
		else
			return;
	
	}

	QString f = QFileDialog::getOpenFileName( previous_dir,filter, this );

	if ( !f.isEmpty() ) // the user gave a file name
	{       		
    		pos = f.findRev('/',f.length()-1);
    			
    		if(pos > 0) // save it for later opens
    		{
    			previous_dir = f.left(pos+1);
    		}
    		
    		QFile qf(f); 
    		
    		if( qf.open(IO_ReadOnly | IO_Translate) == FALSE ) // then the file does not exists
    		{
    			InfoDialog info(this,"Error on File Open");
    			
			info.exec();
		    	return;
    		}
    		    		

		QTextStream qs(&qf);
        
        	if( truss->input_all(&qs) == BAD )
        	{
			delete truss;// bad data will cause repaint to fail
			truss = new truss_model; 
						
        		InfoDialog info(this,"Error on File Read");
       		
        		info.exec();
        		        		       		
		}
        	
        	qf.close();

       
	}
	
	set_limits();
	
	snap_spacing = 0.; // avoid too small spacing when model changes
	
	if(show_nodes == 0) // set show_nodes to yes
		toggle_show_nodes();
				
	repaint();
}

void mywin::input_loads()
{
unsigned int pos;

char filter[10];

	strcpy(filter,"*");
	strcat(filter,LOADS_EXT);

	if( truss->loads() != 0 ) // not empty
	{	
		YesNoDialog yesno(this,"Discard current Loads ?");
		
		if( !yesno.exec() )
			return;
	
	}

	QString f = QFileDialog::getOpenFileName( previous_dir,filter, this );

	if ( !f.isEmpty() ) // the user gave a file name
	{       		
    		pos = f.findRev('/',f.length()-1);
    			
    		if(pos > 0) // save it for later opens
    		{
    			previous_dir = f.left(pos+1);
    		}
    		
    		QFile qf(f); 
    		
    		if( qf.open(IO_ReadOnly | IO_Translate) == FALSE ) // then the file does not exists
    		{
    			InfoDialog info(this,"Error on File Open");
    			
			info.exec();
		    	return;
    		}
    		    		

		QTextStream qs(&qf);
    			
    		int ret;
        	if( (ret=truss->input_loads(&qs)) != OK )
        	{
			if( ret == BAD)
			{			
        			InfoDialog info(this,"Error on Loads File Read");
        			info.exec();
        		}
			if( ret == BAD_NODE)
			{			
        			InfoDialog info(this,"Node Number Mismatch on Loads File Read");
        			info.exec();
        		}
			if( ret == BAD_ELEM)
			{			
        			InfoDialog info(this,"Element Number Mismatch on Loads File Read");
        			info.exec();
        		}
        		        		       		
		}
        	
        	qf.close();
       
	}
	
	repaint();
}

void mywin::input_restraints()
{
unsigned int pos;

char filter[10];

	strcpy(filter,"*");
	strcat(filter,RESTRAINTS_EXT);

	if( truss->restraints() != 0 ) // not empty
	{	
		YesNoDialog yesno(this,"Discard current Restraints ?");
		
		if( !yesno.exec() )
			return;
	
	}

	QString f = QFileDialog::getOpenFileName( previous_dir,filter, this );

	if ( !f.isEmpty() ) // the user gave a file name
	{       		
    		pos = f.findRev('/',f.length()-1);
    			
    		if(pos > 0) // save it for later opens
    		{
    			previous_dir = f.left(pos+1);
    		}
    		
    		QFile qf(f); 
    		
    		if( qf.open(IO_ReadOnly | IO_Translate) == FALSE ) // then the file does not exists
    		{
    			InfoDialog info(this,"Error on File Open");
    			
			info.exec();
		    	return;
    		}
    		    		

		QTextStream qs(&qf);
    			
    		int ret;
        	if( (ret=truss->input_restraints(&qs)) != OK )
        	{
			if( ret == BAD)	
			{		
        			InfoDialog info(this,"Error on Restraints File Read");
        			info.exec();
        		}
			if( ret == BAD_NODE)	
			{		
        			InfoDialog info(this,"Node Number Mismatch on Restraints File Read");
        			info.exec();
        		}
        		        		       		
		}
       	
        	qf.close();
       
	}
	
	repaint();
}

void mywin::help()
{
char text[]="see http://www.aem.umn.edu/courses/aem5515/truss";
	
	InfoDialog info(this,text);

	info.exec();	
}

void mywin::about()
{
char text[100];

	sprintf(text,"Truss by T. Shield, version %s",VERSION);
	InfoDialog info(this,text);
	info.exec();
}


void mywin::toggle_show_nodes()
{
	show_nodes = 1 - show_nodes;
	
	if(show_nodes == 1)
	{
    		//options_menu->setItemChecked(show_nodes_id, TRUE);
    		options_menu->changeItem("Hide nodes",show_nodes_id);
	}
	else
	{
    		//options_menu->setItemChecked(show_nodes_id, FALSE);
    		options_menu->changeItem("Show nodes",show_nodes_id);    		
	}
	
	repaint();
}

void mywin::set_snap()
{

	InputDialog input(this,"Snap spacing (0=none)");
	
	if( input.exec() )
	{
		QString qs(input.get_answer());
		
		bool chk;
		
		float tsnap;
		
		tsnap = qs.toFloat(&chk);
		
		if( chk == TRUE )
		{
			if(tsnap > 0 )
			{
				int n_grids;
		
				n_grids = (int)((xmax-xmin)/tsnap)+(int)((ymax-ymin)/tsnap);
				
				if(n_grids > (width()+height())/2)
				{
					InfoDialog info(this,"Spacing is too small, try again");
					
					info.exec();
					
				}
				else
					snap_spacing = tsnap;
				
			}
			else
				snap_spacing = 0.;
		}
		else
			qApp->beep();
	}
	
	repaint();
}
